/* tools.h: Headerfile to Tools/Library & some macro extensions. --EPM */

/* This file gives a couple of macro extensions as well as function headings
   to the modules of the Tools/Library.  Usually, the documentation is not
   very exhaustive and may be found in the sources. */

/* Note:  +  "cc -DEBUG" is assumed when debugging
          +  global variables are indicated by "extern"
          */

/* to avoid double inclusion */
#ifndef TOOLS_H_SWITCH
#define TOOLS_H_SWITCH 1

/*--------------------------------------------------------------------------*/

#define dynamicDEBUG  0   /* Use dynamicDEBUG == 1 if protocol output and
                             malloc_debug() is desired in module dynamic. */

/*--------------------------------------------------------------------------*/

/* some stuff from standard library /usr/include; see Unix manual pages */
#include <stdio.h>           /* standard IO: FILE ... */
#include <ctype.h>           /* char macros: isalpha() ... */
#include <strings.h>         /* strings: strcmp() ... */
#include <sys/types.h>       /* system types definitions: time_t ... */

/* more standard routines and references */
char * malloc();   /* char * calloc() gives troubles on Suns */
char * realloc();
free();
char * sprintf();
caddr_t sbrk();
extern end, etext, edata;
extern int errno;

/* some constants */
#define MAXINT 2147483647  /* 2^31 - 1 */
#define PI     3.141592654
#define E      2.718281828

/* selected functions from <math.h> to avoid import of stupid gamma
   functions j0(), j1(), y0() and y1(); if you really need them just
   #include <math.h>, that should work, I think */
double fabs(), floor(), ceil(), fmod(), ldexp(), frexp(), sqrt();
double hypot(), atof(), sin(), cos(), tan(), asin(), acos(), atan();
double atan2(), exp(), log(), log10(), pow(), sinh(), cosh(), tanh();

/* capitalized standard type specifiers */
#define INT     int
#define CHAR    char
#define FLOAT   float
#define DOUBLE  double
#define REAL    double
#define BOOL    char

/* other stuff */
#define TRUE  1
#define FALSE 0
#define EOL   '\n'
#define PROC  void
#define IN    /* input parameter */
#define OUT   /* output parameter */

/* some useful macros */
#define odd(X)   ((X) % 2 == 1)
#define even(X)  ((X) % 2 == 0)
#define sign(X)  (((X) > 0) ? 1 : (((X) < 0) ? -1 : 0))
#define iabs(X)  (((X) >= 0) ? X : -(X))
                  /* there were lint problems with abs and INT */

/* human-readable logical operators */
#define and    &&
#define or     ||
#define not    !
#define mod    %
#define _if    /* inline if */
#define _then  ?
#define _else  :

/* nice loop constructs */
#define   upfor(I,S,E)  for (I = (S); I <= (E); I++)
#define downfor(I,S,E)  for (I = (S); I >= (E); I--)
#define until(Q)        while (not (Q));
#define loop            for (;;) 

/* usually, the return values of ?printf() is useless, so: use the following */
#define print        (void) printf
#define fprint       (void) fprintf
#define sprint       (void) sprintf
#define strset(A,B)  (void) sprintf (A, "%s", B)

/* for use in "#define macro(..) macroBegin ... macroEnd" which can be called
   followed by semicolon;  NOTE: Unix cc does not produce larger code, eg:
   isort.o has same size with swap macro with or without macroBegin/End */
#define macroBegin   do {
#define macroEnd     } while (0)

/* Better yet: use "do { blah; blah; blah; } once" instead. */
#define once  while (0)

/*--------------------------------------------------------------------------*/

#ifdef EBUG
#    define version_string(STRING, VERSION)  char STRING[] = "VERSION.D"
#else
#    define version_string(STRING, VERSION)  char STRING[] = "VERSION"
#endif 

/*--------------------------------------------------------------------------*/

/* charbit.c (bit manipulations on CHAR variables: see source) */
extern char charbit_mask__[];
extern char charbit_mask_n[];
#define charbit_on(I,C)  ((charbit_mask__[I] & C) != 0)
#define charbit_s1(I,C)  ((char) (charbit_mask__[I] | C))
#define charbit_s0(I,C)  ((char) (charbit_mask_n[I] & C))

/*--------------------------------------------------------------------------*/

/* dynamic.c (encapsulated dynamic array allocation: see source) */
CHAR * dynamic__new();
CHAR * dynamic__renew();
PROC   dynamic__free();
PROC   dynamic__remark();
PROC   dynamic_info();

#if dynamicDEBUG    /*--------------  debug version  --------------------*/

#define dynamic_allocate(ARRAY,TYPE,SIZE) \
 macroBegin \
  dynamic__remark ("dynamic_allocate", "ARRAY", __FILE__, __LINE__); \
  ARRAY = (TYPE *) dynamic__new ((unsigned INT) (sizeof (TYPE) * (SIZE)), \
                                 FALSE); \
 macroEnd

#define dynamic_allocate_0(ARRAY,TYPE,SIZE) \
 macroBegin \
  dynamic__remark ("dynamic_allocate_0", "ARRAY", __FILE__, __LINE__); \
  ARRAY = (TYPE *) dynamic__new ((unsigned INT) (sizeof (TYPE) * (SIZE)), \
                                 TRUE); \
 macroEnd

#define dynamic_re_allocate(ARRAY,TYPE,NEWSIZE) \
 macroBegin \
  dynamic__remark ("dynamic_re_allocate", "ARRAY", __FILE__, __LINE__); \
  ARRAY = (TYPE *) dynamic__renew ((CHAR *) ARRAY, \
                                 (unsigned INT) (sizeof (TYPE) * (NEWSIZE))); \
 macroEnd

#define dynamic_dispose(ARRAY) \
 macroBegin \
  dynamic__remark ("dynamic_dispose", "ARRAY", __FILE__, __LINE__); \
  if (ARRAY) { dynamic__free ((CHAR *) ARRAY); ARRAY = NULL; } \
 macroEnd

#else    /*------------------ normal version -------------------------------*/

#define dynamic_allocate(ARRAY,TYPE,SIZE) \
 ARRAY = (TYPE *) dynamic__new ((unsigned INT) (sizeof (TYPE) * (SIZE)), FALSE)

#define dynamic_allocate_0(ARRAY,TYPE,SIZE) \
 ARRAY = (TYPE *) dynamic__new ((unsigned INT) (sizeof (TYPE) * (SIZE)), TRUE)

#define dynamic_re_allocate(ARRAY,TYPE,NEWSIZE) \
 ARRAY = (TYPE *) dynamic__renew ((CHAR *) ARRAY, \
                                  (unsigned INT) (sizeof (TYPE) * (NEWSIZE)))

#define dynamic_dispose(ARRAY) \
 macroBegin \
  if (ARRAY) { dynamic__free ((CHAR *) ARRAY); ARRAY = NULL; } \
 macroEnd

#endif

/*--------------------------------------------------------------------------*/

/* isort.c (inplace (insertion) sorting of very small lists: see source) */
INT isort2();
INT isort3();
INT isort4p();
INT isort4();
INT isort5p();

/*--------------------------------------------------------------------------*/

/* error.c (error stop and error hook: see source) */
PROC error();

/*--------------------------------------------------------------------------*/

/* assert(EX) is a macro that indicates expression EX is expected to be true
   (1) at this point in the program.  It calls the error() routine a diagnostic
   comment when EX is false.  Compiling *without* the -DEBUG option deletes
   assert() from the program.  
   NOTE: This is a slightly modified /usr/include/assert.h and it isn't
   used too often since we have error().  However, note the __FILE__ and
   __LINE__ constants understood by cc preprocessor. */
#ifdef EBUG
# define assert(EX) \
   if (not (EX))  error ("Assertion failed in file \"%s\" at line %d.", \
                         __FILE__, __LINE__)
#else
# define assert(ex) /* nothing */
#endif

/*--------------------------------------------------------------------------*/

/* readline.c (read line from file: see source) */
INT readlinef();
#define readline  (void) readlinef

/*--------------------------------------------------------------------------*/

/* get.c (system time etc: see source) */
INT  get_user_seconds();
PROC get_host_name();
PROC get_date_and_time();
PROC get_time_of_day();
REAL get_standard_time();
INT  get_some_seed();

/*--------------------------------------------------------------------------*/

#endif
