/* lia.c: module lia: long integer arithmetic --EPM */
/* Version: - w/o unpacking 
            - w/ local double precision (long INT) multiplication
            - w/ BASE == power of 2
              ==> shift instead of div  and bitwise or instead of modulo
            - w/ length of IN parameters arbitrary */

/*LINTLIBRARY*/

#define SPM 1
/* switch to enable the lia_single_precision_mul option; set to 0 or 1 */

#include "tools.h"
#include "lia.h"

static INT mul_loops = 0, mul_calls = 0;
static INT add_loops = 0, add_calls = 0;
static INT len = -1;      /* lia-object: a[0], a[1],..., a[len] */

#if SPM
  BOOL lia_single_precision_mul = FALSE;
  /* If this option is TRUE, then lia_mul does multiplication only w.r.t. lia-
     digits [0] & [1] w/o any overflow checking.  Note: This option is sort of
     inofficial and not included in lia.h; however, it's still globally
     accessible.  The whole thing works only if swich SPM != 0. */
#endif

/*--------------------------------------------------------------------------*/

/* beta == 15 */

#define BASE         32768             /* 2^15 */
#define DBASE        1073741824        /* BASE * BASE; lia-digit < DBASE */

#define modBASE     & 32767      /* a modBASE == a (bitwise AND) (BASE - 1) */
#define divBASE     >> 15        /* a divBASE == a (shift right) log2 BASE */
#define timesBASE   << 15
#define modDBASE    & 1073741823
#define divDBASE    >> 30

/*--------------------------------------------------------------------------*/

#define div2    >> 1
#define mod2    & 1
#define times2  << 1

/*--------------------------------------------------------------------------*/

/* Buffer space for lia_chars() etc; not static since also used in lia2.c */

#define CBUFLEN 2000
CHAR lia__cbuf[CBUFLEN];
INT lia__cptr = 0;
CHAR lia__cbover[] = "buffer overflow; use lia_reset";
PROC lia__cput(), lia__dput();

/*--------------------------------------------------------------------------*/

PROC lia_setlength (length)
     IN INT length;
     /* lia initialization; sets maximal lenght of lia_*() results;
        lia-object: LIA x[length] = x[0], x[1], ..., x[length-1] */
{
  len = length - 1;
}

/*--------------------------------------------------------------------------*/

INT lia_getlength ()
     /* inverse to lia_lenght (L); returns L */
{
  return (len + 1);
}

/*--------------------------------------------------------------------------*/

PROC lia_counters_output (file)
     FILE * file;
     /* prints counters to file */
{
  fprint (file, "* lia counters\n");
  fprint (file, "%12d . lia_mul calls\n", mul_calls);
  fprint (file, "%12d . lia_mul loops\n", mul_loops);
  fprint (file, "%12d . lia_add calls\n", add_calls);
  fprint (file, "%12d . lia_add loops\n", add_loops);
}

/*--------------------------------------------------------------------------*/

PROC lia_reset ()
        /* resets the lia__cbuf[] */
{
  lia__cptr = 0;
}

/*--------------------------------------------------------------------------*/

PROC lia_load (longi, shorti)
     OUT LIA  longi[];
     IN  long INT shorti;
     /* initializes a long integer lia-object: longi := shorti; 
        assumes longi[0..2], at least */
{
  if (len < 2)
    error ("lia_load: length < 2");
  longi[1] = (LIA) iabs (shorti) modDBASE;
  longi[2] = (LIA) iabs (shorti) divDBASE;
  longi[0] = _if (longi[2] == 0) _then 2 _else 4;
  if (shorti < 0)
    longi[0] ++;
}

/*--------------------------------------------------------------------------*/

PROC lia_strload (longi, string, frmt)
     OUT LIA longi[];
     IN CHAR string[], frmt[];
     /* loads hex or dec string (frmt == "%x" or "%d") into longi[0..len];
        longi := string;  uses lia__cbuf[] */
{
  LIA basis[3], ziffer[3];
  LIA * hilfe = (LIA *) (& lia__cbuf[lia__cptr]);
  CHAR digit[2];
  INT i = 0;
  BOOL negative = FALSE;
  long INT d;
  lia__cptr = lia__cptr + sizeof (LIA) * (len + 1);
  if (lia__cptr > CBUFLEN)
    error ("lia_strload: %s", lia__cbover);
  if (frmt[0] != '%')
    error ("lia_strload: wrong frmt");
  switch (frmt[1])
    {
      case 'd': lia_load (basis, 10L); break;
      case 'x': lia_load (basis, 16L); break;
      default:  error ("lia_strload: wrong frmt %c", frmt[1]);
    }
  lia_load (longi, 0L);
  digit[1] = 0;
  while (string[i] == ' ')
    i ++;
  if (string[i] == '-')
    {
      negative = TRUE;  i ++;
    }
  else if (string[i] == '+')
    i ++;
  while (digit[0] = string[i++])
    {
      (void) sscanf (digit, frmt, &d);
      lia_load (ziffer, d);
      lia_mul (hilfe, longi, basis);
      lia_add (longi, hilfe, ziffer);   /* longi := basis * longi + ziffer */
    }
  if (negative)
    lia_chs (longi);
}

/*--------------------------------------------------------------------------*/

PROC lia_fput (file, longi)
     FILE *file;
     IN LIA longi[];
     /* prints _internal_ representation of longi to file */
{
  INT i;
  fprint (file, "%c:", _if (lia_sign (longi) == -1) _then '-' _else '+');
  downfor (i, longi[0] div2, 1)
    fprint (file, "%ld:", longi[i]);
}

/*--------------------------------------------------------------------------*/

REAL lia_real (longi)
     IN LIA longi[];
     /* converts lia-object to real;  not too fast! */
{
  INT i;
  REAL b = 1.0, r = 0.0;
  upfor (i, 1, longi[0] div2)
    {
      r += longi[i] * b;
      b *= (REAL) DBASE;
    }
  return (r * (REAL) lia_sign (longi));
}

/*--------------------------------------------------------------------------*/

CHAR * lia_chars (longi, decimal)
     IN LIA longi[];
     IN BOOL decimal;
     /* Returns a pointer to a CHAR string containing the decimal or
        hexadecimal representation of longi.  Usually this should work within
        a print statement using more than one lia_chars() calls.
        However, the bufferspace lia__cbuf[] is limitied by CBUFLEN and it
        is advisible to call lia_reset() once in a while, or even
        after each print statement.  If you want to save the decimal
        string you'll have to copy it into one of your own string
        variables since some other lia procedures might destroy the buffer. */
{
  INT p;
  LIA * lbuf = (LIA *) (& lia__cbuf[lia__cptr]);
  lia_let (lbuf, longi);
  p = lia__cptr = lia__cptr + sizeof (LIA) * ((longi[0] div2) + 1);
  if (lia_sign (longi) == 0)
    {
      lia__cput ('+0');  lia__cput ('\0');
      return (& (lia__cbuf[p]));
    }
  if (lia_sign (lbuf) == -1)
    {
      lia__cput ('-');  lia_chs (lbuf);
    }
  else
    lia__cput ('+');
  lia__dput (lbuf, decimal);   /* print and destroy lbuf[] to lia__cbuf[] */
  lia__cput ('\0');
  while (lia__cbuf[++p] == '0')
    lia__cbuf[p] = lia__cbuf[p-1];   /* get rid of +00xxxxx etc */
  return (& (lia__cbuf[p-1]));
}

/*--------------------------------------------------------------------------*/

PROC lia__dput (longi, decimal)
     IN OUT LIA longi[];
     IN BOOL decimal;
     /* puts decimal (or hexadecimal) representation of longi[] into cbbuf[];
        code is sort of weird (recursive!) to avoid additional buffer space;
        NOTE that longi[] will destroyed; additional assumption: longi[] > 0 */
{
  INT part;
  CHAR string[5];
  lia_sdiv (longi, &part, longi, _if (decimal) _then 1000 _else 0x1000);
  sprint (string, _if (decimal) _then "%03d" _else "%03x", part);
  if ((longi[0] > 2) or (longi[1] > 0))
    lia__dput (longi, decimal);
  lia__cput (string[0]);
  lia__cput (string[1]);
  lia__cput (string[2]);
}

/*--------------------------------------------------------------------------*/

PROC lia__cput (c)
     IN CHAR c;
     /* adds c to lia__cbuf[] and checks for overflow */
{
  lia__cbuf[lia__cptr] = c;
  lia__cptr ++;
  if (lia__cptr > CBUFLEN)
    error ("lia_*: %s", lia__cbover);
}

/*--------------------------------------------------------------------------*/

PROC lia_let (long1, long2)
     OUT LIA long1[];
     IN  LIA long2[];
     /* long integer assignment: long1 := long2 */
{
  register INT i;
  upfor (i, 0, long2[0] div2)
    long1[i] = long2[i];
}

/*--------------------------------------------------------------------------*/

PROC lia_chs (longi)
     IN OUT LIA longi[];
     /* change of sign:  longi := -longi  if long != 0 */
{
  if (((longi[0] div2) > 1) or (longi[1] != 0))
    if (odd (longi[0]))
      longi[0] --;
    else
      longi[0] ++;
}

/*--------------------------------------------------------------------------*/

INT lia_sign (longi)
     IN LIA longi[];
     /* returns the sign (-1, 0, or +1) of a long integer longi */
{
  if ((longi[0] == 2) and (longi[1] == 0))
    return (0);
  else if (odd (longi[0]))
    return (-1);
  else
    return (1);
}

/*--------------------------------------------------------------------------*/

INT lia_high (longi)
     IN LIA longi[];
     /* returns the number of the highest lia_digit used in longi */
{
  return (longi[0] div2);
}

/*--------------------------------------------------------------------------*/

LIA lia_digit (longi, d)
     IN LIA longi[];
     IN INT d;
     /* returns longi[d] without any further checking */
{
 return (longi[d]);
}
   
/*--------------------------------------------------------------------------*/

BOOL lia_eq (long1, long2)
     IN LIA long1[], long2[];
     /* returns long1 == long2 */
{
  register INT i;
  register BOOL res = FALSE;
  if (long1[0] == long2[0])
    {
      i = long1[0] div2;
      while ((i > 0) and (long1[i] == long2[i]))
        i --;
      res = (i <= 0);
    }
  return (res);
}

/*--------------------------------------------------------------------------*/

BOOL lia_le (long1, long2)
     IN LIA long1[], long2[];
     /* returns long1 < long2 */
{
  unsigned short INT sign1, sign2;
  sign1 = long1[0] mod2;
  sign2 = long2[0] mod2;
  return (((sign1 != sign2) and (sign1 == 1))
          or
          ((sign1 == sign2)
           and
           (((sign1 == 0) and lia_ple (long1, long2))
            or
            ((sign1 == 1) and lia_ple (long2, long1)))));
}       

/*--------------------------------------------------------------------------*/

BOOL lia_ple (long1, long2)
     IN LIA long1[], long2[];
     /* returns abs (long1) < abs (long2) */
{
  short INT last1, last2;
  BOOL  fin, res;
  last1 = long1[0] div2;
  last2 = long2[0] div2;
  if (last1 != last2)
    res = (last1 < last2);
  else
    {
      do
        {
          fin = TRUE;
          if (long1[last1] != long2[last1])
            res = (long1[last1] < long2[last1]);
          else
            {
               last1 --;
               fin = FALSE;
            }
        } until ((last1 < 0) or fin);
    }
  if (last1 < 0)
    res = FALSE;
  return (res);
}

/*--------------------------------------------------------------------------*/

BOOL lia_leq (long1, long2)
     IN LIA long1[], long2[];
     /* returns long1 <= long2 */
{
  return (lia_le (long1, long2) or lia_eq (long1, long2));
}

/*--------------------------------------------------------------------------*/

PROC lia_add (longi, long1, long2)
     OUT LIA longi[];
     IN  LIA long1[], long2[];
     /* long integer addition: longi := long1 + long2 */
{
  unsigned short INT sign1;
  add_calls ++;
  sign1 = long1[0] mod2;
  if (sign1 == (long2[0] mod2))
    {
      lia_padd (longi, long1, long2);
      longi[0] += sign1;
    }
  else if (sign1 == 0)
    {
      if (lia_ple (long1, long2))
        {
          lia_psub (longi, long2, long1);
          longi[0] ++;
        }
      else
        lia_psub (longi, long1, long2);
    }
  else if (lia_ple (long1, long2))
    lia_psub (longi, long2, long1);
  else
    {
      lia_psub (longi, long1, long2);
      if ((longi[0] != 0) or (longi[1] != 0))
        longi[0] ++;
    } 
  if ((longi[0] == 3) and (longi[1] == 0))         /* -0 correction */
    /* don't really know why/whether still needed, but it works now */
    longi[0] = 2;
}

/*--------------------------------------------------------------------------*/

PROC lia_sub (longi, long1, long2)
     OUT LIA longi[];
     IN  LIA long1[], long2[];
     /* long integer subtraction: longi := long1 - long2 */
{
  lia_chs (long2);
  lia_add (longi, long1, long2);
  lia_chs (long2);  /* undo first line */
}

/*--------------------------------------------------------------------------*/

PROC lia_padd (longi, long1, long2)
     OUT LIA longi[];
     IN  LIA long1[], long2[];
     /* long := abs (long1) + abs (long2) */
{
  register LIA sum, carry = 0;
  register INT i, max = long1[0] div2, len2 = long2[0] div2;
  if (max < len2)
    lia_padd (longi, long2, long1);
  else
    { /* lia_high (long1) >= lia_high (long2[0]) */
      upfor (i, 1, len2)
        {
          sum = long1[i] + long2[i] + carry;
          longi[i] = sum modDBASE;
          carry = sum divDBASE;
        }
      upfor (i, len2 + 1, max)
        {
          sum = long1[i] + 0 + carry;
          longi[i] = sum modDBASE;
          carry = sum divDBASE;
        }
      longi[0] = (unsigned) max times2;
      add_loops += max;
      if (carry)
        {
          longi[max+1] = 1;
          longi[0] += 2;
          if (max == len)
            error ("lia_add or lia_sub: overflow");
        }
    }
}

/*--------------------------------------------------------------------------*/

PROC lia_psub (longi, long1, long2)
     OUT LIA longi[];
     IN  LIA long1[], long2[];
     /* longi := abs (long1) - abs (long2) 
        provided that abs (long1) >= abs (long2) */
{
  register LIA a, b, carry = 0;
  register INT i, max = long1[0] div2, len2 = long2[0] div2;
  longi[0] = 1;
  upfor (i, (long2[0] div2) + 1, max) long2[i] = 0;
  add_loops += max;
  upfor (i, 1, len2)
    {
      a = long1[i];  b = long2[i] + carry;
      if (b > a)
        {
          longi[i] = ((a + (DBASE - b)) modDBASE);  carry = 1;
        }
      else
        {
          longi[i] = a - b;  carry = 0;
        }
      if (longi[i] > 0)
        longi[0] = i;
    }
  upfor (i, len2 + 1, max)
    {
      a = long1[i];  /* b == 0 + carry */
      if (carry > a)
        {
          longi[i] = ((a + (DBASE - carry)) modDBASE);  carry = 1;
        }
      else
        {
          longi[i] = a - carry;  carry = 0;
        }
      if (longi[i] > 0)
        longi[0] = i;
    }
  longi[0] = longi[0] times2;
}

/*--------------------------------------------------------------------------*/

PROC lia_mul (longi, long1, long2)
     OUT LIA longi[];
     IN  LIA long1[], long2[];
     /* long integer multiplication: longi := long1 * long2
        version: w/ local double precision (long INT) multiplication */
{
  register LIA a1, a2, b1, b2, x1, x2, carry;
  register INT i, j, h;
  register unsigned short INT ind1, ind2;
  mul_calls ++;
  ind1 = long1[0] div2;
  ind2 = long2[0] div2;
  if (ind1 + ind2 - 1 > len)
    error ("lia_mul: overflow (a priory)");
  if (((ind1 == 1) and (long1[1] == 0)) or ((ind2 == 1) and (long2[1] == 0)))
    /* either long1 is zero or long2 is zero; in other words:
       (not (is_non_zero (long1) and is_non_zero (long2))) is true */
    lia_load (longi, 0L);
  else
    {
      mul_loops += ind1 * ind2;
      upfor (i, 1, ind1 + ind2 - 1)
        longi[i] = 0;
      longi[0] = (((ind1 + ind2 - 1) times2)
                  + ((long1[0] mod2) + ((long2[0] mod2)) mod2));
      upfor (i, 1, ind1)
        {
          carry = 0;
          h = i;
          upfor (j, 1, ind2)
            {
              /* compute (hi,lo) = long1[i] * long2[j] + carry + longi[h]
                 using double precision (long INT) multiplication;
                 note: h == i + j - 1 */
              /* break up into half digits (a1,a2) and (b1,b2) */
              a1 = long1[i] divBASE;
              a2 = long1[i] modBASE;
              b1 = long2[j] divBASE;
              b2 = long2[j] modBASE;
              /* (x1,x2) = (a1,a2) * (b1,b2) */
              x1 = a1 * b1;
              x2 = a2 * b2;
              a1 = a1 * b2;
              a2 = a2 * b1;
              /* note: (a1,a2) below used as (a1b2,a2b1) */
              x1 +=  (a1 divBASE) + (a2 divBASE);
              x2 += ((a1 modBASE) + (a2 modBASE)) timesBASE;
              /* move higher bits of x2 to x1 since only 2 overflow bits */
              x1 += (x2 divDBASE);
              x2  = (x2 modDBASE);
              /* (x1,x2) = (x1,x2) + carry + longi[h] */
              x2 += carry + longi[h];  /* note: order is important */
              x1 += (x2 divDBASE);     /* note: x1 is always < DBASE */
              x2 = (x2 modDBASE);
              /* store in digit and carry */
              longi[h] = x2;
              carry = x1;
              h ++;
            }
          if (i < ind1)
            longi[h] = carry;
          else if (carry > 0)
            {
              if (/*i + ind2*/ h > len)
                error ("lia_mul: overflow (the last carry)");
              longi[h] = carry;
            }
        }
      if (carry > 0)
        longi[0] += 2; /* since last carry > 0 */
    }
}

/*--------------------------------------------------------------------------*/

PROC lia_sdiv (result, remainder, longi, shorti)
     OUT LIA result[];
     OUT INT * remainder;
     IN LIA longi[];
     IN INT shorti;
     /* does simple divison  longi / shorti  with  0 < shorti < BASE
        NOTE: result and longi may denote the same LIA object */
{
  register LIA i, r, s, t, e, f, u, v;
  BOOL zero = TRUE /* for the time being */, negative = odd (longi[0]);
  s = DBASE / shorti;
  t = DBASE mod shorti;
  r = 0;
  downfor (i, longi[0] div2, 1)
    {
      e = longi[i] / shorti;
      f = longi[i] mod shorti;
      u = r * s + e;
      v = r * t + f;  /* overflow in v == ... is possible when shorti > BASE */
      result[i] = u + v / shorti;
      r = v mod shorti;
    }
  *remainder = _if (negative) _then -r _else r;
  downfor (i, longi[0] div2, 1) if (result[i] > 0)
    {
      result[0] = i times2;
      zero = FALSE;
      break;
    }
  if (negative)
    result[0] ++;
  if (zero)
    result[0] = 2;
}
