#
#  Issue a command prompt and set the marks for use with command
#  history
#
proc _key(Prompt) {} {
  global _prompt
  .commands mark set insert end-1c
  .commands insert insert $_prompt {prompt output}
  .commands insert insert " " {prompt input}
  .commands tag add input insert end
  .commands tag add newline end-1c
  .commands mark set prompt insert
  .commands mark gravity prompt left
  .commands mark set history insert
  .commands mark gravity history left
}

#
#  Check if an event is in an output region
#
proc _key(OutputEvent) {code i} {
  if [regexp {output|prompt} [.commands tag names $i]] {return -code $code}
}
#
#  Check if an event is at the end of an input region
#
proc _key(NewlineEvent) {code i} {
  if [regexp {newline} [.commands tag names $i]] {return -code $code}
}

#
#  Check that the maximum number of lines is not exceeded
#  Remove extra lines (up to the next prompt)
#
proc _key(CheckLines) {} {
  global _prompt _maxlines
  set n [string length $_prompt]
  while {[.commands index end] > $_maxlines} {
    set line [.commands tag nextrange prompt 0.0+$n\c]
    .commands delete 0.0 [lindex $line 0]
  }
}

#
#  Return the text of the previous line in the history, and move the
#  history marker
#
proc _key(PrevHistory) {} {
  if [.commands compare history == 1.0] {return {}}
  if [.commands compare {history linestart} == 1.0] {
    set first 1.0
    set last 1.0
  } else {
    set first [_key(LastPrompt) {history linestart -1c}]
    set last [_key(NextNewline) $first]
    if [.commands compare $first == history] {set first 1.0; set last 1.0}
  }
  .commands mark set history $first
  return [.commands get $first $last]
}

#
#  Return the text of the next line in the history, and move the
#  history marker
#
proc _key(NextHistory) {} {
  if {[.commands compare history == prompt]} {return {}}
  set first [lindex [.commands tag nextrange prompt history] 1]
  .commands mark set history $first
  if {[.commands compare $first == prompt]} {return {}}
  set last [_key(NextNewline) $first]
  return [.commands get $first $last]
}

#
#  Check if the history line is the first one in the buffer
#
proc _key(IsFirst) {} {
  return [.commands compare {history linestart} == \
            [lindex [.commands tag nextrange prompt 0.0] 0]]
}

#
#  Check if the history line is the last one in the buffer
#
proc _key(IsLast) {} {
  return [expr {[.commands tag nextrange prompt history] == ""}]
}

#
#  If there are no more history lines, ring the bell
#  Otherwise
#    If we are moving from the input line, save the current input
#    Get the previous history line
#
proc _key(CopyPrevHistory) {} {
  global _key
  if [_key(IsFirst)] bell else {
    if [_key(IsLast)] {set _key(last) [_key(GetInput) insert]}
    _key(CopyInput) [_key(PrevHistory)] 1.0
  }
}

#
#  If there are no more history lines, ring the bell
#  Otherwise
#    Copy the next history line
#    If we are now at the bottom, bring back the original input that
#      we saved when going up
#
proc _key(CopyNextHistory) {} {
  global _key
  if [_key(IsLast)] bell else {
    _key(CopyInput) [_key(NextHistory)] 1.0
    if [_key(IsLast)] {_key(CopyInput) $_key(last) 1.0}
  }
}

#
#  Find the index of the start of the previous input line
#
proc _key(LastPrompt) {i} {
  set i [.commands index "$i linestart"]
  while {(![regexp "prompt" [.commands tag names $i]]) &&
	 ([.commands compare $i > 1.0])} {
    set i [.commands index $i-1line]
  }
  return [lindex [.commands tag nextrange prompt $i] 1]
}

#
#  Find the end of the specified input line
#
proc _key(NextNewline) {i} {
  return [lindex [.commands tag nextrange newline $i] 0]
}

#
#  Returns the input line at the specified location
#
proc _key(GetInput) {i} {
  set first [_key(LastPrompt) $i]
  set last [_key(NextNewline) $first]
  return [.commands get $first $last]
}

#
#  Copys a command to the prompt area if it is not already there
#  and sets the insert to after the command
#
proc _key(CopyInput) {cmd i} {
  if {[.commands compare $i < {prompt linestart}]} {
    .commands delete prompt {end -1c}
    .commands insert prompt $cmd input
  }
  .commands mark set insert {end -1c}
  .commands see insert
}



#
#  Set up key bindings for use by the shells
#
bind _key(map) <Return> {
  if {!([%W compare insert >= prompt] && [%W compare insert != {end - 1c}] &&
        [%W compare {prompt lineend} != {end - 1c}])} {
    _cmd(Invoke)
    break
  }
}
bind _key(map) <KP_Enter> {_cmd(Invoke); break}

#
#  Only do character insertion and pasting in input areas
#
bind _key(map) <Any-KeyPress>    {_key(OutputEvent) break insert}
bind _key(map) <ButtonRelease-2> {_key(OutputEvent) break @%x,%y}
bind _key(map) <Insert>          {_key(OutputEvent) break insert}

#
#  Allow these commands to operate anywhere
#
bind _key(map) <Any-Up>       { }
bind _key(map) <Any-Down>     { }
bind _key(map) <Any-Prior>    { }
bind _key(map) <Any-Next>     { }
bind _key(map) <Any-Home>     { }
bind _key(map) <Any-End>      { }
bind _key(map) <Any-Select>   { }
bind _key(map) <Meta-greater> { }
bind _key(map) <Meta-w>       { }
bind _key(map) <Control-backslash> { }
bind _key(map) <Any-Control-space> { }
bind _key(map) <Control-e>    { }

#
#  Make ESCAPE work like META for these
#
bind _key(map) <Escape><greater> [concat [bind Text <Meta-greater>] "; break"]
bind _key(map) <Escape>w [concat [bind Text <Meta-w>] "; break"]

#
#  Check that these commands don't run off the end of an input area
#
bind _key(map) <Any-Left> {
  _key(OutputEvent) return insert
  _key(OutputEvent) break {insert-1c}
}
bind _key(map) <Control-b> {
  _key(OutputEvent) return insert
  _key(OutputEvent) break {insert-1c}
}
bind _key(map) <Meta-b> {
  _key(OutputEvent) return insert
  _key(OutputEvent) break {insert-1c wordstart}
}
bind _key(map) <Escape>b {
  _key(OutputEvent) return insert
  _key(OutputEvent) break {insert-1c wordstart}
}
bind _key(map) <Any-Right> {
  _key(OutputEvent) return insert
  _key(NewlineEvent) break insert
}
bind _key(map) <Control-f> {
  _key(OutputEvent) return insert
  _key(NewlineEvent) break insert
}
bind _key(map) <Meta-f> {
  _key(OutputEvent) return insert
  _key(NewlineEvent) break insert
}
bind _key(map) <Escape>f {
  _key(OutputEvent) return insert
  _key(NewlineEvent) break insert
}
bind _key(map) <Control-a> {
  _key(OutputEvent) return insert
  if [regexp {prompt} [%W tag names {insert linestart}]] {
    tkTextSetCursor %W [_key(LastPrompt) insert]
    break
  }
}

#
#  The "top of the file" should be after the first prompt
#
bind _key(map) <Control-Home> {
  _key(OutputEvent) return insert
  %W mark set insert 1.0
  tkTextSetCursor %W [lindex [%W tag nextrange prompt insert] 1] 
  break
}
bind _key(map) <Meta-less> {
  _key(OutputEvent) return insert
  %W mark set insert 1.0
  tkTextSetCursor %W [lindex [%W tag nextrange prompt insert] 1] 
  break
}
bind _key(map) <Escape><less> {
  _key(OutputEvent) return insert
  %W mark set insert 1.0
  tkTextSetCursor %W [lindex [%W tag nextrange prompt insert] 1] 
  break
}

#
#  Control-\ only selects the current input line
#
bind _key(map) <Control-slash> {
  _key(OutputEvent) return insert
  %W tag add sel [_key(LastPrompt) insert] [_key(NextNewline) insert]
  break
}

#
#  Make sure these commands are entirely within the input area
#
bind _key(map) <Control-d> {
  _key(OutputEvent) break insert
  _key(NewlineEvent) break insert
}
bind _key(map) <Delete> {
  _key(OutputEvent) break insert
  _key(NewlineEvent) break insert
}
bind _key(map) <Meta-d> {
  _key(OutputEvent) break insert
  _key(OutputEvent) break {insert wordend}
}
bind _key(map) <Escape>d {
  _key(OutputEvent) break insert
  _key(OutputEvent) break {insert wordend}
}
bind _key(map) <Control-h> {
  _key(OutputEvent) break insert
  _key(OutputEvent) break {insert -1c}
}
bind _key(map) <BackSpace> {
  _key(OutputEvent) break insert
  _key(OutputEvent) break {insert -1c}
}
bind _key(map) <Meta-BackSpace> {
  _key(OutputEvent) break insert
  _key(OutputEvent) break {insert -1c wordstart}
}
bind _key(map) <Escape><BackSpace> {
  _key(OutputEvent) break insert
  _key(OutputEvent) break {insert -1c wordstart}
}
bind _key(map) <Meta-Delete> {
  _key(OutputEvent) break insert
  _key(OutputEvent) break {insert -1c wordstart}
}
bind _key(map) <Escape><Delete> {
  _key(OutputEvent) break insert
  _key(OutputEvent) break {insert -1c wordstart}
}
bind _key(map) <Control-t> {
  _key(OutputEvent) break insert
  _key(OutputEvent) break {insert -1c}
  if {[%W compare insert == {insert lineend}]} \
    {_key(OutputEvent) break {insert -2c}}
}

#
#  Have Up and Down scroll through the history if they are pressed
#  at the top or bottom of the input line (kind of a wierd effect
#  but basically the right idea)
#
bind _key(hmap) <Control-p> {
  if {[%W compare {insert linestart} == {prompt linestart}]} \
    {_key(CopyPrevHistory); break}
}
bind _key(hmap) <Up> {
  if {[%W compare {insert linestart} == {prompt linestart}]} \
    {_key(CopyPrevHistory); break}
}
bind _key(hmap) <Control-n> {
  if {[%W compare {insert lineend} == {end-1c}]} \
    {_key(CopyNextHistory); break}
}
bind _key(hmap) <Down> {
  if {[%W compare {insert lineend} == {end-1c}]} \
    {_key(CopyNextHistory); break}
}

#
#  Reset history pointer for any other keys
#
bind _key(hmap) <Any-Key> {
  .commands mark set history insert
  .commands mark gravity history left
  set _key(last) ""
}

#
#  Allow shifted versions to avoid the history mechanism
#
bind _key(map) <Control-P> {
  tkTextSetCursor %W [tkTextUpDownLine %W -1]
  break
}

bind _key(map) <Control-N> {
  tkTextSetCursor %W [tkTextUpDownLine %W 1]
  break
}

bind _key(map) <Control-B> {
  tkTextSetCursor %W [%W index {insert - 1c}]
  break
}

bind _key(map) <Control-F> {
  tkTextSetCursor %W [%W index {insert + 1c}]
  break
}

#
#  Erase to the beginning of the input line and save the data
#  to the clipboard (for retrieval by CTRL-Y).
#
bind _key(map) <Control-u> {
  _key(OutputEvent) break insert
  clipboard clear -displayof %W
  clipboard append -displayof %W -- \
      [%W get [_key(LastPrompt) insert] insert]
  %W delete [_key(LastPrompt) insert] insert
}

#
#  If there is a selected range, CTRL-W will cut it to the clipboard,
#  Otherwise it will backward erase a word (for Stuart)
#
bind _key(map) <Control-w> {
  _key(OutputEvent) break insert
  if {[%W tag nextrange sel 1.0 end] == ""} {
    _key(OutputEvent) break {insert -1c wordstart}
    %W delete {insert -1c wordstart} insert
    break
  }
}

#
#  CTRL-R and ESC-P will search backward for the previous command that
#  starts with the characters to the left of the cursor
#
bind _key(hmap) <Control-r> \
  {if {[%W compare prompt < insert]} {_key(ctrl-r); break}}
bind _key(hmap) <Escape>p \
  {if {[%W compare prompt < insert]} {_key(ctrl-r); break}}
bind _key(hmap) <Escape> { }

proc _key(ctrl-r) {} {
  global _key
  if [_key(IsLast)] {set _key(last) [_key(GetInput) insert]}
  set pattern [.commands get prompt insert]
  set len [expr [string length $pattern] - 1]
  while {[.commands compare history != 1.0]} {
    set match [_key(PrevHistory)]
    if {[string compare [string range $match 0 $len] $pattern] == 0} {
      _key(CopyInput) $match 1.0
      .commands mark set insert "prompt + [expr $len + 1] chars"
      .commands see insert
      return
    }
  }
  bell
}

bindtags .commands {_key(hmap) _key(map) .commands Text . all}
_ctrlK(Bind) .commands
