Require button.tcl
Require label.tcl
Require menu.tcl
Require name.tcl
Require option.tcl

_option(Add) cedit.bbox.color.width 1
_option(Add) cedit.bbox.color.height 1
_option(Add) cedit.bbox.color.padX 1m
_option(Add) cedit.bbox.color.padY 1


ooRoot Instance _ColorEdit {
  Var {colors {}}
  Var {action {}}
  Var {centered 0}
  Var id count rgb object
  Var {clip {}}

  Method Close {} {
    grab release .cedit
    wm withdraw .cedit
    update idletasks
  }
  Method Cancel {} {Self Close}

  Method OK {} {
    Vars action
    Self Close
    if {$action != ""} {
      set cmd $action
      set action {}
      uplevel \#0 $cmd
    }
  }

  Method Color {} {_SelectRGB Request [val rgb] "[Self] SetColor"}

  Method New {} \
    {_name(Request) "New color name:" "" {_ColorEdit _New $_name(name)}}
  Method _New {name} {
    global _color
    Vars id colors
    if {$id < [llength $colors]} {
      set color [lreplace [lindex $colors $id] 2 2 $name]
    } else {
      set color [list {.5 .5 .5} $_color(count) $name]
    }
    set colors [linsert $colors $id $color]
    .cedit.colors insert $id $name
    Self Select $id
  }

  Method Delete {} {
    Vars colors id
    if {[Confirm "Really delete color '[lindex [lindex $colors $id] 2]'?"]} \
	return
    set colors [lreplace $colors $id $id]
    .cedit.colors delete $id
    Self Select $id
  }

  Method Rename {} {
    Vars colors id
    set color [lindex [lindex $colors $id] 2]
    _name(Request) "Rename color '$color' as:" $color \
	{_ColorEdit _Rename $_name(name)}
  }
  Method _Rename {name} {
    Vars colors id
    set color [lindex $colors $id]
    set colors [lreplace $colors $id $id [lreplace $color 2 2 $name]]
    .cedit.colors delete $id
    .cedit.colors insert $id $name
  }

  Method Copy {} {
    set [var clip] [val colors]
    .cedit.bbox.paste configure -state normal
  }
  Method Paste {} {Self SetColorList [val clip]}

  Method Defaults {} {
    global _color
    Self Copy
    Self SetColorList $_color(default)
  }

  Method Request {} {
    Vars action colors id clip object
    set object [_Current]
    set action "[Self] _Request"
    Self SetColorList [$object Color get color-list]
    if {$clip == ""} {.cedit.bbox.paste configure -state disabled}
    Self Center
    if [winfo ismapped .cedit] {wm iconify .cedit}
    wm deiconify .cedit
    if {[grab current .] != ""} {grab release [grab current .]}
    grab set .cedit
  }
  Method _Request {} {
    global _color
    Vars colors object
    Self Save
    _File Changed 1
    set _color(color-list) $colors
    $object Color set color-list $colors
    $object Color MenuCheck
    $object AutoColor
  }

  Method SetColorList {list} {
    Vars object colors
    set colors $list
    .cedit.colors delete 0 end
    .cedit.colors(title) configure -text "Colors for '[file tail $object]':"
    foreach color $colors {.cedit.colors insert end [lindex $color 2]}
    .cedit.colors insert end " -- "
    Self Select 0
  }

  Method Center {} {
    Vars centered
    if {!$centered} {
      set x [expr [winfo screenwidth .cedit]/2 - [winfo reqwidth .cedit]/2 - \
		  [winfo vrootx [winfo parent .cedit]]]
      set y [expr [winfo screenheight .cedit]/3 - [winfo reqheight .cedit]/2 -\
		  [winfo vrooty [winfo parent .cedit]]]
      wm geom .cedit +$x+$y
      set centered 1
    }
  }

  Method Save {} {
    Vars id count colors rgb
    if {$id != "" && $id < [llength $colors]} {
      Self CheckCount
      set colors [lreplace $colors $id $id \
        [list $rgb $count [lindex [lindex $colors $id] 2]]]
    }
  }

  Method Select {i} {
    Vars id count colors
    set id $i
    .cedit.colors selection clear 0 end
    .cedit.colors selection set $id
    if {$id < [llength $colors]} {
      set count [lindex [lindex $colors $id] 1]
      Self SetColor [lindex [lindex $colors $id] 0]
      .cedit.bbox.color configure -state normal
      .cedit.count configure -state normal
      .cedit.bbox.rename configure -state normal
      .cedit.bbox.delete configure -state normal
    } else {
      set count ""
      .cedit.bbox.color configure -background "\#D9D9D9" -state disabled
      .cedit.count configure -state disabled
      .cedit.bbox.rename configure -state disabled
      .cedit.bbox.delete configure -state disabled
    }
  }

  Method SetColor {color} {
    Vars rgb
    set rgb $color
    set R [expr {int([lindex $rgb 0]*255)}]
    set G [expr {int([lindex $rgb 1]*255)}]
    set B [expr {int([lindex $rgb 2]*255)}]
    .cedit.bbox.color configure -background [format "\#%02X%02X%02X" $R $G $B]
    set R [expr {$R+int((255-$R)/2)}]
    set G [expr {$G+int((255-$G)/2)}]
    set B [expr {$B+int((255-$B)/2)}]
    .cedit.bbox.color configure \
       -activebackground [format "\#%02X%02X%02X" $R $G $B]
  }

  Method CheckCount {} {
    Vars count id
    if {![regexp {^[0-9]+$} $count]} {
      .cedit.colors selection clear 0 end
      .cedit.colors selection set $id
      Error "Count must be an integer"
    }
  }

  Method Hit {w} {
    Vars id
    set i [$w curselection]
    if {$i != $id && $i != ""} {
      Self Save
      Self Select $i
    }
  }
}

toplevel .cedit
wm withdraw .cedit
wm title .cedit "$_program(id): Color Editor"
wm minsize .cedit 200 200

_menu(Scrollable) .cedit colors "Colors:" 25 9
pack configure .cedit.colors(box) -side left -expand 1 -fill both -padx 0
_button(Box) .cedit bbox bottom {
  {Action ok      "_OK"        {_ColorEdit OK}}
  {Action cancel  "_Cancel"    {_ColorEdit Cancel}}
  {Action default "Defaults"   {_ColorEdit Defaults}}
  {Action paste   "Paste List"      {_ColorEdit Paste}}
  {Action copy    "Copy List"       {_ColorEdit Copy}}
  {Action delete  "_Delete..." {_ColorEdit Delete}}
  {Action rename  "_Rename..." {_ColorEdit Rename}}
  {Action new     "_New..."    {_ColorEdit New}}
} right
pack configure .cedit.bbox -fill none

button .cedit.bbox.color -command {_ColorEdit Color}
pack .cedit.bbox.color -side left -padx 2 -pady 2

entry .cedit.count -width 3 -justify center \
  -textvariable [_ColorEdit var count]
pack .cedit.count -in .cedit.bbox -side right -expand 1 -fill x \
  -padx 2 -ipady 1

bind .cedit <Return>   {_button(Invoke) .cedit.bbox.ok}
bind .cedit <KP_Enter> {_button(Invoke) .cedit.bbox.ok}
bind .cedit <Escape>   {_button(Invoke) .cedit.bbox.cancel}

bind _cedit(keymap) <Any-KeyPress> {_ColorEdit Hit %W}
bind _cedit(keymap) <Any-1> {_ColorEdit Hit %W}
bind _cedit(keymap) <Any-2> {_ColorEdit Hit %W}
bind _cedit(keymap) <Any-3> {_ColorEdit Hit %W}
bind _cedit(keymap) <Any-ButtonRelease-1> {_ColorEdit Hit %W}
bind _cedit(keymap) <Any-ButtonRelease-2> {_ColorEdit Hit %W}
bind _cedit(keymap) <Any-ButtonRelease-3> {_ColorEdit Hit %W}
bind _cedit(keymap) <Any-B1-Motion> {_ColorEdit Hit %W}

bindtags .cedit.colors {.cedit.colors Listbox _cedit(keymap) .cedit all}
