#######################
#
#  Vector math function
#

set _vector(epsilon) 1.0e-6

proc let {variable value args} {
  set len [llength $variable]
  if {$len == 1} {
    if {[llength $value] == 1} {
      uplevel [list set $variable] "\[expr $value $args\]"
    } else {
      foreach x $value {lappend X [uplevel "expr $x"]}
      uplevel [list set $variable $X]
    }
  } else {
    set value [concat $value $args]
    if {[llength $value] != $len} \
      {return -code error "Incorrect number of assignments"}
    for {set i 0} {$i < $len} {incr i} \
      {uplevel [list set [lindex $variable $i]] "\[expr [lindex $value $i]\]"}
  }
}

proc + {X {Y ""}} {
  if {$Y == ""} {return $X}
  set len [llength $X]
  if {$len != [llength $Y]} \
    {return -code error "Can't add matrices of different sizes"}
  for {set i 0} {$i < $len} {incr i} \
    {lappend Z [uplevel "expr ([lindex $X $i]) + ([lindex $Y $i])"]}
  return $Z
}

proc - {X {Y ""}} {
  set len [llength $X]
  if {$Y == ""} {
    foreach x $X {lappend Z [uplevel "expr -($x)"]}
  } else {
    if {$len != [llength $Y]} \
      {return -code error "Can't subtract matrices of different sizes"}
    for {set i 0} {$i < $len} {incr i} \
      {lappend Z [uplevel "expr ([lindex $X $i]) - ([lindex $Y $i])"]}
  }
  return $Z
}

proc / {X Y} {
  if {[llength $Y] == 2 && [llength $X] == 2 && [info procs C/] != ""} {
    return [C/ $X $Y]
  } else {
    if {[llength $Y] != 1} {return -code error "Division must be by a scalar"}
    set c [uplevel "expr $Y"]; set Z {}
    if {$c == 0} {return -code error "Division by zero"}
    foreach x $X {lappend Z [uplevel "expr ($x)/double($c)"]}
    return $Z
  }
}

proc Dot {X Y} {
  set len [llength $X]
  if {$len != [llength $Y]} \
    {return -code error "Can't dot vectors of different lengths"}
  set a 0
  for {set i 0} {$i < $len} {incr i} \
    {set a [expr $a + [uplevel "expr ([lindex $X $i]) * ([lindex $Y $i])"]]}
  return $a
}

proc >< {X Y} {
  if {[llength $X] != 3 || [llength $Y] != 3} \
    {return -code error "Cross product only allowed in dimension 3"}
  set X1 [uplevel "expr [lindex $X 0]"]
  set X2 [uplevel "expr [lindex $X 1]"]
  set X3 [uplevel "expr [lindex $X 2]"]
  set Y1 [uplevel "expr [lindex $Y 0]"]
  set Y2 [uplevel "expr [lindex $Y 1]"]
  set Y3 [uplevel "expr [lindex $Y 2]"]
  return [list [expr $X2*$Y3-$X3*$Y2] \
	       [expr $X3*$Y1-$X1*$Y3] \
	       [expr $X1*$Y2-$X2*$Y1]]
}

proc Norm {X} {
  set s 0
  foreach x $X {set s [uplevel "expr $s + ($x)*($x)"]}
  return [expr sqrt($s)]
}

proc Unit {X} {
  set s 0
  foreach x $X {set s [uplevel "expr $s + ($x)*($x)"]}
  set n [expr sqrt($s)]
  if {$n == 0.0} {return $X}
  foreach x $X {lappend U [uplevel "expr ($x)/double($n)"]}
  return $U
}

proc Linear {a X b Y} {
  set len [llength $X]
  if {$len != [llength $Y]} \
    {return -code error \
	 "Linear products must be of vectors of the same dimension"}
  if {[llength $a] != 1 || [llength $b] != 1} \
    {return -code error "Linear products must have scalar coefficients"}
  for {set i 0} {$i < $len} {incr i} \
    {lappend Z [uplevel "expr ($a)*([lindex $X $i]) + ($b)*([lindex $Y $i])"]}
  return $Z
}

proc * {X Y} {
  set Xl [llength $X]
  set Yl [llength $Y]
  set Z {}
  if {$Xl == 1} {
    foreach y $Y {lappend Z [uplevel "expr ($X)*($y)"]}
  } elseif {$Yl == 1} {
    foreach x $X {lappend Z [uplevel "expr ($x)*($Y)"]}
  } elseif {$Xl == $Yl} {
    if {$Xl == 2 && [info procs {C\*}] != ""} {set Z [C* $X $Y]} else {
      set n [expr int(sqrt($Xl))]
      if {$n*$n != $Xl} {return -code error "Matrices must be square"}
      for {set i 0} {$i < $n} {incr i; lappend Z $R} {
	for {set j 0; set R ""} {$j < $n} {incr j; lappend R $m} {
	  set m 0; set x [expr $i * $n]; set y $j;
	  for {set k 0} {$k < $n} {incr k; incr x; incr y $n} \
	      {set m [uplevel "expr $m + ([lindex $X $x])*([lindex $Y $y])"]}
	}
      }
      set Z [join $Z \n]
    }
  } else {
    set n [expr int(sqrt($Xl))]
    if {$n*$n != $Xl} {return -code error "Matrix must be square"}
    if {$n == $Yl+1} {lappend Y 1} elseif {$n != $Yl} \
      {return -code error "Matrix and vector dimensions don't match"}
    for {set i 0; set k 0} {$i < $n} {incr i; lappend Z $m} {
      for {set j 0; set m 0} {$j < $n} {incr j; incr k} \
	{set m [uplevel "expr $m + ([lindex $X $k])*([lindex $Y $j])"]}
    }
    if {$n == $Yl+1} {set Z [lreplace $Z end end]}
  }
  return $Z
}

proc == {X Y} {
  if {[llength $X] != [llength $Y]} {return 0}
  for {set i [expr [llength $X]-1]} {$i >= 0} {incr i -1} {
    if {[uplevel "expr [lindex $X $i]"] != [uplevel "expr [lindex $Y $i]"]} \
       {return 0}
  }
  return 1
}

proc != {X Y} {expr ! [uplevel [list == $X $Y]]}

proc === {X Y} {
  global _vector
  if {[llength $X] != [llength $Y]} {return 0}
  for {set i [expr [llength $X]-1]} {$i >= 0} {incr i -1} {
    if {[uplevel "expr abs(([lindex $X $i]) - ([lindex $Y $i]))"] >
	  $_vector(epsilon)} {return 0}
  }
  return 1
}

proc !== {X Y} {expr ! [uplevel [list === $X $Y]]}

proc >> {X Y} {
  set n [uplevel "expr $Y"]
  if {$n != int($n)} {return -code error "Shift must be by an integer amount"}
  set len [llength $X]
  while {$n > 0} {incr n -$len}
  while {$n <= -$len} {incr n $len}
  if {$n == 0} {return $X}
  set n [expr -$n]
  set X [concat [lrange $X $n end] [lrange $X 0 [expr $n-1]]]
  return $X
}

proc << {X Y} {
  set n [uplevel "expr $Y"]
  if {$n != int($n)} {return -code error "Shift must be by an integer amount"}
  set len [llength $X]
  while {$n < 0} {incr n $len}
  while {$n >= $len} {incr n -$len}
  if {$n == 0} {return $X}
  set X [concat [lrange $X $n end] [lrange $X 0 [expr $n-1]]]
  return $X
}


set _axis(x) 1
set _axis(y) 2
set _axis(z) 3
set _axis(w) 4
set _axis(X) 1
set _axis(Y) 2
set _axis(Z) 3
set _axis(W) 4

proc Coord {X Y} {
  global _axis
  set Z {}
  foreach x $X {
    if [info exists _axis($x)] {set x $_axis($x)}
    incr x -1
    lappend Z [uplevel "expr [lindex $Y $x]"]
  }
  return $Z
}
proc Extract {Y X} [info body Coord]


proc Transpose {M} {
  set Ml [llength $M]
  set d [expr int(sqrt($Ml))]
  if {$d*$d != $Ml} {return -code error "Matrices must be square"}
  for {set i 0; set k 0; set T ""} {$i < $d} {incr i; set k $i; lappend T $R} {
    for {set j 0; set R ""} {$j < $d} {incr j; incr k $d} \
	{lappend R [lindex $M $k]}
  }
  return [join $T \n]
}
