#! /usr/local/bin/wish

#
#  These variables control the look of the TclShell.
#  You can source shell.tcl and then modify them before
#  calling TclShell in order to change the look
#

set _tclshell(version)      1.5
set _tclshell(fontsize)	    14
set _tclshell(input.font)   "*-courier-medium-r-normal--$_tclshell(fontsize)-*"
set _tclshell(input.color)  black
set _tclshell(prompt.font)  "*-courier-medium-r-normal--$_tclshell(fontsize)-*"
set _tclshell(prompt.color) grey50
set _tclshell(prompt)       "%"
set _tclshell(output.font)  "*-courier-medium-o-normal--$_tclshell(fontsize)-*"
set _tclshell(output.color) black
set _tclshell(error.font)   "*-courier-medium-o-normal--$_tclshell(fontsize)-*"
set _tclshell(error.color)  red3
set _tclshell(select.bg)    grey95
set _tclshell(select.fg)    black
set _tclshell(select.bw)    0
set _tclshell(bg)           grey80
set _tclshell(fg)           black
set _tclshell(width)        50
set _tclshell(height)       15
set _tclshell(maxlines)     1000
set _tclshell(basename)     .tclshell
set _tclshell(exit)         exit

#
#  Main entry point:
#
#     TclShell ?title? ?w? ?h?
#
#  Opens a TclShell window for interactive commands.  You can
#  Have as many of these as you want open at once.  To destroy
#  one, use the command "exit" or press Control-d
#
proc TclShell {{title "Tcl Shell"} {w ""} {h ""}} \
  {_tclshell(Toplevel) $title $w $h}


#
#  Create a new toplevel window and add a TCL shell to it
#  Returns a pointer to the toplevel window
#
proc _tclshell(Toplevel) {{title "Tcl Shell"} {w ""} {h ""}} {
  global _tclshell
  set name $_tclshell(basename)
  if {[winfo exists $_tclshell(basename)]} {
    set i 2
    while {[winfo exists $name$i]} {incr i}
    set name "$name$i"
    set title "$title \#$i"
  }
  toplevel $name -bg $_tclshell(bg)
  wm title $name $title
  wm minsize $name 5 2

  _tclshell(Create) $name $w $h
  focus $name.text
  return $name
}

#
#  Adds a TclShell to an existing frame widget
#  Returns a pointer to the text widget
#
proc _tclshell(Create) {frame {w ""} {h ""}} {
  global _tclshell

  if {[winfo exists $frame.text]} {return $frame.text}
  if {$w == ""} {set w $_tclshell(width)}
  if {$h == ""} {set h $_tclshell(height)}

  text $frame.text -relief sunken -bd 2 -setgrid true \
      -width $w -height $h -bg $_tclshell(bg) -fg $_tclshell(fg) \
      -yscrollcommand "$frame.scroll set" \
      -selectborderwidth $_tclshell(select.bw) \
      -selectbackground $_tclshell(select.bg) \
      -selectforeground $_tclshell(select.fg) \
      -font $_tclshell(input.font)
  scrollbar $frame.scroll -command "$frame.text yview" \
      -bg $_tclshell(bg) 
  pack $frame.scroll -side right -fill both
  pack $frame.text -side left -expand 1 -fill both

  bindtags $frame.text [linsert [bindtags $frame.text] 0 \
      _tclshell(ctrl-k.map) _tclshell(hist.map) _tclshell(keymap)]

  $frame.text tag configure output \
      -font $_tclshell(output.font) \
      -foreground $_tclshell(output.color)
  $frame.text tag configure input \
      -font $_tclshell(input.font) \
      -foreground $_tclshell(input.color)
  $frame.text tag configure prompt \
      -font $_tclshell(prompt.font) \
      -foreground $_tclshell(prompt.color)
  $frame.text tag configure error \
      -font $_tclshell(error.font) \
      -foreground $_tclshell(error.color)

  $frame.text insert insert "\[tclshell version $_tclshell(version)\]\n" output
  _tclshell(Prompt) $frame.text
  return $frame.text
}

#
#  Destroy a command window (if it is a top level)
#
proc _tclshell(Destroy) {w code} {
  set frame [file rootname $w]
  if {$frame != "" && $frame == [winfo toplevel $frame]} {
    destroy $frame
    if {[regexp {^shell.tcl( |$)} [winfo name .]]} exit
    return -code $code
  }
}

#
#  Check if an event is in an output region
#
proc _tclshell(OutputEvent) {code w i} {
  if [regexp {output|prompt} [$w tag names $i]] {return -code $code}
}
#
#  Check if an event is at the end of an input region
#
proc _tclshell(NewlineEvent) {code w i} {
  if [regexp {newline} [$w tag names $i]] {return -code $code}
}

#
#  Check that the maximum number of lines is not exceeded
#  Remove extra lines (up to the next prompt)
#
proc _tclshell(CheckLines) {w} {
  global _tclshell
  set n [string length $_tclshell(prompt)]
  while {[$w index end] > $_tclshell(maxlines)} {
    set line [$w tag nextrange prompt 0.0+$n\c]
    $w delete 0.0 [lindex $line 0]
  }
}

#
#  Print a prompt and tag it correctly
#  Set the prompt and history marks to their proper places
#
proc _tclshell(Prompt) {w} {
  global _tclshell
  $w mark set insert end-1c
  $w insert insert $_tclshell(prompt) {prompt output}
  $w insert insert " " {prompt input}
  $w tag add input insert end
  $w tag add newline end-1c
  $w mark set prompt insert
  $w mark gravity prompt left
  $w mark set history insert
  $w mark gravity history left
}

#
#  Return the text of the previous line in the history, and move the
#  history marker
#
proc _tclshell(PrevHistory) {w} {
  if [$w compare history == 1.0] {return {}}
  if [$w compare {history linestart} == 1.0] {
    set first 1.0
    set last 1.0
  } else {
    set first [_tclshell(LastPrompt) $w {history linestart -1c}]
    set last [_tclshell(NextNewline) $w $first]
    if [$w compare $first == history] {set first 1.0; set last 1.0}
  }
  $w mark set history $first
  return [$w get $first $last]
}

#
#  Return the text of the next line in the history, and move the
#  history marker
#
proc _tclshell(NextHistory) {w} {
  if {[$w compare history == prompt]} {return {}}
  set first [lindex [$w tag nextrange prompt history] 1]
  $w mark set history $first
  if {[$w compare $first == prompt]} {return {}}
  set last [_tclshell(NextNewline) $w $first]
  return [$w get $first $last]
}

#
#  Check if the history line is the first one in the buffer
#
proc _tclshell(IsFirst) {w} {
  return [$w compare {history linestart} == \
            [lindex [$w tag nextrange prompt 0.0] 0]]
}

#
#  Check if the history line is the last one in the buffer
#
proc _tclshell(IsLast) {w} {
  return [expr {[$w tag nextrange prompt history] == ""}]
}

#
#  If there are no more history lines, ring the bell
#  Otherwise
#    If we are moving from the input line, save the current input
#    Get the previous history line
#
proc _tclshell(CopyPrevHistory) {w} {
  global _tclshell
  if [_tclshell(IsFirst) $w] bell else {
    if [_tclshell(IsLast) $w] \
      {set _tclshell(last-$w) [_tclshell(GetInput) $w insert]}
    _tclshell(CopyInput) $w [_tclshell(PrevHistory) $w] 1.0
  }
}

#
#  If there are no more history lines, ring the bell
#  Otherwise
#    Copy the next history line
#    If we are now at the bottom, bring back the original input that
#      we saved when going up
#
proc _tclshell(CopyNextHistory) {w} {
  global _tclshell
  if [_tclshell(IsLast) $w] bell else {
    _tclshell(CopyInput) $w [_tclshell(NextHistory) $w] 1.0
    if [_tclshell(IsLast) $w] {_tclshell(CopyInput) $w $_tclshell(last-$w) 1.0}
  }
}

#
#  Find the index of the start of the previous input line
#
proc _tclshell(LastPrompt) {w i} {
  set i [$w index "$i linestart"]
  while {(![regexp "prompt" [$w tag names $i]]) &&
	 ([$w compare $i > 1.0])} {
    set i [$w index $i-1line]
  }
  return [lindex [$w tag nextrange prompt $i] 1]
}

#
#  Find the end of the specified input line
#
proc _tclshell(NextNewline) {w i} {
  return [lindex [$w tag nextrange newline $i] 0]
}

#
#  Returns the input line at the specified location
#
proc _tclshell(GetInput) {w i} {
  set first [_tclshell(LastPrompt) $w $i]
  set last [_tclshell(NextNewline) $w $first]
  return [$w get $first $last]
}

#
#  Copys a command to the prompt area if it is not already there
#  and sets the insert to after the command
#
proc _tclshell(CopyInput) {w cmd i} {
  if {[$w compare $i < {prompt linestart}]} {
    $w delete prompt {end -1c}
    $w insert prompt $cmd input
  }
  $w mark set insert {end -1c}
  $w see insert
}

#
#  Executes a command line and prints the results
#  Only complete commands are executed, and if the command is "exit"
#  then the shell is destroyed
#
proc _tclshell(Invoke) {w} {
  global _tclshell
  set cmd [_tclshell(GetInput) $w insert]
  if [info complete $cmd] {
    _tclshell(CopyInput) $w $cmd insert
    $w insert insert \n {input newline}
    $w see insert; update idletasks
    if {$cmd == $_tclshell(exit)} {_tclshell(Destroy) $w return}
    set result [catch [list uplevel \#0 $cmd] msg]
    if {$result == 0} {
      if {$msg != ""} {$w insert insert $msg\n output}
    } else {
      $w insert insert "Error: $msg\n" {error output}
    }
    _tclshell(CheckLines) $w
    _tclshell(Prompt) $w
  } else {
    $w insert insert \n input
  }
  $w see insert
}

#
#  Set up key bindings for use by the shells
#
bind _tclshell(keymap) <Return> {
  if {!([%W compare insert >= prompt] && [%W compare insert != {end - 1c}] &&
        [%W compare {prompt lineend} != {end - 1c}])} {
    _tclshell(Invoke) %W
    break
  }
}
bind _tclshell(keymap) <KP_Enter> {_tclshell(Invoke) %W; break}

#
#  Only do character insertion and pasting in input areas
#
bind _tclshell(keymap) <Any-KeyPress> \
    {_tclshell(OutputEvent) break %W insert}
bind _tclshell(keymap) <ButtonRelease-2> \
    {_tclshell(OutputEvent) break %W @%x,%y}
bind _tclshell(keymap) <Insert> \
    {_tclshell(OutputEvent) break %W insert}

#
#  Allow these commands to operate anywhere
#
bind _tclshell(keymap) <Any-Up>       { }
bind _tclshell(keymap) <Any-Down>     { }
bind _tclshell(keymap) <Any-Prior>    { }
bind _tclshell(keymap) <Any-Next>     { }
bind _tclshell(keymap) <Any-Home>     { }
bind _tclshell(keymap) <Any-End>      { }
bind _tclshell(keymap) <Any-Select>   { }
bind _tclshell(keymap) <Meta-greater> { }
bind _tclshell(keymap) <Meta-w>       { }
bind _tclshell(keymap) <Control-backslash> { }
bind _tclshell(keymap) <Any-Control-space> { }
bind _tclshell(keymap) <Control-e>    { }

#
#  Make ESCAPE work like META for these
#
bind _tclshell(keymap) <Escape><greater> \
  [concat [bind Text <Meta-greater>] "; break"]
bind _tclshell(keymap) <Escape>w [concat [bind Text <Meta-w>] "; break"]

#
#  Check that these commands don't run off the end of an input area
#
bind _tclshell(keymap) <Any-Left> {
  _tclshell(OutputEvent) return %W insert
  _tclshell(OutputEvent) break %W {insert-1c}
}
bind _tclshell(keymap) <Control-b> {
  _tclshell(OutputEvent) return %W insert
  _tclshell(OutputEvent) break %W {insert-1c}
}
bind _tclshell(keymap) <Meta-b> {
  _tclshell(OutputEvent) return %W insert
  _tclshell(OutputEvent) break %W {insert-1c wordstart}
}
bind _tclshell(keymap) <Escape>b {
  _tclshell(OutputEvent) return %W insert
  _tclshell(OutputEvent) break %W {insert-1c wordstart}
}
bind _tclshell(keymap) <Any-Right> {
  _tclshell(OutputEvent) return %W insert
  _tclshell(NewlineEvent) break %W insert
}
bind _tclshell(keymap) <Control-f> {
  _tclshell(OutputEvent) return %W insert
  _tclshell(NewlineEvent) break %W insert
}
bind _tclshell(keymap) <Meta-f> {
  _tclshell(OutputEvent) return %W insert
  _tclshell(NewlineEvent) break %W insert
}
bind _tclshell(keymap) <Escape>f {
  _tclshell(OutputEvent) return %W insert
  _tclshell(NewlineEvent) break %W insert
}
bind _tclshell(keymap) <Control-a> {
  _tclshell(OutputEvent) return %W insert
  if [regexp {prompt} [%W tag names {insert linestart}]] {
    tkTextSetCursor %W [_tclshell(LastPrompt) %W insert]
    break
  }
}

#
#  The "top of the file" should be after the first prompt
#
bind _tclshell(keymap) <Control-Home> {
  _tclshell(OutputEvent) return %W insert
  %W mark set insert 1.0
  tkTextSetCursor %W [lindex [%W tag nextrange prompt insert] 1] 
  break
}
bind _tclshell(keymap) <Meta-less> {
  _tclshell(OutputEvent) return %W insert
  %W mark set insert 1.0
  tkTextSetCursor %W [lindex [%W tag nextrange prompt insert] 1] 
  break
}
bind _tclshell(keymap) <Escape><less> {
  _tclshell(OutputEvent) return %W insert
  %W mark set insert 1.0
  tkTextSetCursor %W [lindex [%W tag nextrange prompt insert] 1] 
  break
}

#
#  Control-\ only selects the current input line
#
bind _tclshell(keymap) <Control-slash> {
  _tclshell(OutputEvent) return %W insert
  %W tag add sel \
     [_tclshell(LastPrompt) %W insert] [_tclshell(NextNewline) %W insert]
  break
}

#
#  Control-d will destroy the shell if it is pressed when the input
#  area is blank
#
bind _tclshell(keymap) <Control-d> {
  _tclshell(OutputEvent) break %W insert
  if {[%W compare prompt == {end - 1c}] && [%W compare prompt == insert]} {
    _tclshell(Destroy) %W break
  }
  _tclshell(NewlineEvent) break %W insert
}

#
#  Make sure these commands are entirely within the input area
#
bind _tclshell(keymap) <Delete> {
  _tclshell(OutputEvent) break %W insert
  _tclshell(NewlineEvent) break %W insert
}
bind _tclshell(keymap) <Meta-d> {
  _tclshell(OutputEvent) break %W insert
  _tclshell(OutputEvent) break %W {insert wordend}
}
bind _tclshell(keymap) <Escape>d {
  _tclshell(OutputEvent) break %W insert
  _tclshell(OutputEvent) break %W {insert wordend}
}
bind _tclshell(keymap) <Control-h> {
  _tclshell(OutputEvent) break %W insert
  _tclshell(OutputEvent) break %W {insert -1c}
}
bind _tclshell(keymap) <BackSpace> {
  _tclshell(OutputEvent) break %W insert
  _tclshell(OutputEvent) break %W {insert -1c}
}
bind _tclshell(keymap) <Meta-BackSpace> {
  _tclshell(OutputEvent) break %W insert
  _tclshell(OutputEvent) break %W {insert -1c wordstart}
}
bind _tclshell(keymap) <Escape><BackSpace> {
  _tclshell(OutputEvent) break %W insert
  _tclshell(OutputEvent) break %W {insert -1c wordstart}
}
bind _tclshell(keymap) <Meta-Delete> {
  _tclshell(OutputEvent) break %W insert
  _tclshell(OutputEvent) break %W {insert -1c wordstart}
}
bind _tclshell(keymap) <Escape><Delete> {
  _tclshell(OutputEvent) break %W insert
  _tclshell(OutputEvent) break %W {insert -1c wordstart}
}
bind _tclshell(keymap) <Control-t> {
  _tclshell(OutputEvent) break %W insert
  _tclshell(OutputEvent) break %W {insert -1c}
  if {[%W compare insert == {insert lineend}]} {
    _tclshell(OutputEvent) break %W {insert -2c}
  }
}

#
#  Have Up and Down scroll through the history if they are pressed
#  at the top or bottom of the input line (kind of a wierd effect
#  but basically the right idea)
#
bind _tclshell(hist.map) <Control-p> {
  if {[%W compare {insert linestart} == {prompt linestart}]} \
    {_tclshell(CopyPrevHistory) %W; break}
}
bind _tclshell(hist.map) <Up> {
  if {[%W compare {insert linestart} == {prompt linestart}]} \
    {_tclshell(CopyPrevHistory) %W; break}
}
bind _tclshell(hist.map) <Control-n> {
  if {[%W compare {insert lineend} == {end-1c}]} \
    {_tclshell(CopyNextHistory) %W; break}
}
bind _tclshell(hist.map) <Down> {
  if {[%W compare {insert lineend} == {end-1c}]} \
    {_tclshell(CopyNextHistory) %W; break}
}

#
#  Reset history pointer for any other keys
#
bind _tclshell(hist.map) <Any-Key> {
  %W mark set history insert
  %W mark gravity history left
  set _key(last-%W) ""
}

#
#  Allow shifted versions to avoid the history mechanism
#
bind _tclshell(keymap) <Control-P> {
  tkTextSetCursor %W [tkTextUpDownLine %W -1]
  break
}

bind _tclshell(keymap) <Control-N> {
  tkTextSetCursor %W [tkTextUpDownLine %W 1]
  break
}

bind _tclshell(keymap) <Control-B> {
  tkTextSetCursor %W [%W index {insert - 1c}]
  break
}

bind _tclshell(keymap) <Control-F> {
  tkTextSetCursor %W [%W index {insert + 1c}]
  break
}

#
#  A special keymap for CTRL-K allows us to tell if the last thing pressed
#  was a CTRL-K (so that we add to the clipboard).
#
bind _tclshell(ctrl-k.map) <Any-KeyPress> {set _tclshell(ctrl-k.add) 0}
bind _tclshell(ctrl-k.map) <Any-1> {set _tclshell(ctrl-k.add) 0}
bind _tclshell(ctrl-k.map) <Any-2> {set _tclshell(ctrl-k.add) 0}
bind _tclshell(ctrl-k.map) <Any-3> {set _tclshell(ctrl-k.add) 0}
bind _tclshell(ctrl-k.map) <Enter> {set _tclshell(ctrl-k.add) 0}
bind _tclshell(ctrl-k.map) <Control-k> { }
bind _tclshell(keymap) <Control-backslash> {clipboard clear -displayof %W}

set _tclshell(ctrl-k.add) 0

bind _tclshell(keymap) <Control-k> {
  _tclshell(OutputEvent) break %W insert
  _tclshell(NewlineEvent) break %W insert
  _tclshell(ctrl-k) %W
}

#
#  For a CTRL-K, clear the clipboard if this is the first one
#  Add the rest of the line to the clipboard and record that
#  we have been pressing CTRL-K.
#
#   CTRL-Y gets this data back.
#
proc _tclshell(ctrl-k) {w} {
  global _tclshell

  if {! $_tclshell(ctrl-k.add)} {clipboard clear -displayof $w}
  set text [$w get insert {insert lineend}]
  if {[$w compare insert == {insert lineend}]} {set text \n}
  catch {clipboard append -displayof $w -- $text}
  set _tclshell(ctrl-k.add) 1
}

#
#  Erase to the beginning of the input line and save the data
#  to the clipboard (for retrieval by CTRL-Y).
#
bind _tclshell(keymap) <Control-u> {
  _tclshell(OutputEvent) break %W insert
  clipboard clear -displayof %W
  clipboard append -displayof %W -- \
      [%W get [_tclshell(LastPrompt) %W insert] insert]
  %W delete [_tclshell(LastPrompt) %W insert] insert
}

#
#  If there is a selected range, CTRL-W will cut it to the clipboard,
#  Otherwise it will backward erase a word (for Stuart)
#
bind _tclshell(keymap) <Control-w> {
  _tclshell(OutputEvent) break %W insert
  if {[%W tag nextrange sel 1.0 end] == ""} {
    _tclshell(OutputEvent) break %W {insert -1c wordstart}
    %W delete {insert -1c wordstart} insert
    break
  }
}

#
#  CTRL-R and ESC-P will search backward for the previous command that
#  starts with the characters to the left of the cursor
#
bind _tclshell(hist.map) <Control-r> \
  {if {[%W compare prompt < insert]} {_tclshell(ctrl-r) %W; break}}
bind _tclshell(hist.map) <Escape>p \
  {if {[%W compare prompt < insert]} {_tclshell(ctrl-r) %W; break}}
bind _tclshell(hist.map) <Escape> { }


proc _tclshell(ctrl-r) {w} {
  global _tclshell
  if [_tclshell(IsLast) $w] \
    {set _tclshell(last-$w) [_tclshell(GetInput) $w insert]}
  set pattern [$w get prompt insert]
  set len [expr [string length $pattern] - 1]
  while {[$w compare history != 1.0]} {
    set match [_tclshell(PrevHistory) $w]
    if {[string compare [string range $match 0 $len] $pattern] == 0} {
      _tclshell(CopyInput) $w $match 1.0
      $w mark set insert "prompt + [expr $len + 1] chars"
      $w see insert
      return
    }
  }
  bell
}


#
#  If we run this script interactively, open the shell immediately
#

if {[regexp {^shell.tcl( |$)} [winfo name .]]} {
  TclShell
  wm withdraw .
}
