######################################################################
#
#  fname.tcl
#
#  Implements a fancy file requester widget.  The widget includes
#  a list of file names and allows you to navigate the file system
#  wither by typing names or by pointing and clicking.	The files
#  can be filtered to show only certain ones, and the filter can be
#  changed on the fly by the user.
#
#  The widget includes buttons for making new directories and for
#  deleting file or (empty) directories.  It also has a menu that
#  lists recently-visited directories, so it is not too hard to work
#  with several directories at once.
#
#  There are two main entry points:
#
#    _fname(NewFile) title name action ?ext? ?dir?
#    _fname(OldFile) title name action ?ext? ?dir?
#
#  where "title" is the message to display just above the name entry,
#  "name" is the default file name, action is the script to run when
#  the "Open" or "Save" button is pressed, "ext" is a file extension
#  that will be used to form the initial filter (*.ext), and "dir"
#  is the initial directory where the file requester will start (if
#  "name" includes directory components, it will be combined with
#  "dir" to make a full directory path).
#
#  When the user presses the "Open" or "Save" button the "action" script
#  will run.  The script can contain occurances of "%N", and these will
#  be replaced by the file name of the file specified by the user.  The
#  file name is the full file name, including a complete directory path.
#
#  You can also get the current file by calling _fname(Name).
#
######################################################################
#
#  who	  when	   what
#  ---	  ----	   ----
#
#  dpvc   8/97	   Added options value to make customizing easier
#
#  dpvc   6/97	   Added grab when dialog window is open.
#
#  dpvc   6/96     Added feature where directory name will show "..."
#                  when it is too long to fit in the menu button.
#
#  dpvc	  4/96	   wrote it.
#
######################################################################
#
#  version:  1.2
#

#
#  Set some variables if they aren't there:
#    _program(id) specifies the name of the _program that is running
#
if ![info exists _program(id)]  {set _program(id) [info script]}
if ![info exists env(PWD)]     {set env(PWD) [exec pwd]}

set _fname(title)  "";	     # The message just above the file name entry
set _fname(dir)	   "";	     # The current directory being displayed
set _fname(dname)  "";	     # The text for the directory button
set _fname(filter) "*";	     # The current file-name filter
set _fname(name)   "";	     # The contents of the name entry
set _fname(showAll)    0;    # True when all files are to be shown
set _fname(scanRate)   75;   # How fast to scroll when dragging the list
set _fname(autoSelect) 1;    # True if files in the list are to be selected
			     #	automatically when characters are typed


set _fname(action)   {}
set _fname(centered) 0
set _fname(recent)   {}
set _fname(afterID)  {}

set _fname(level)    30;     # "option add" priority level

######################################################################
#
#  Make the top-level window and it widgets
#
toplevel .fname
wm withdraw .fname
wm title .fname "$_program(id): File Name Requester"
wm minsize .fname 300 125

#
#  Set the default sizes, fonts, etc.  These can be modified by the user
#  via the xrdb database or the .Xdefaults file
#
option add *fname.mbar*relief		groove	$_fname(level)
option add *fname.mbar*borderWidth	3	$_fname(level)
option add *fname.mbar*menu.relief	raise	$_fname(level)
option add *fname.mbar*menu.borderWidth	2	$_fname(level)
option add *fname.mbar*menu.activeBorderWidth 0	$_fname(level)
option add *fname.mbar*menu.activeBackground white $_fname(level)
option add *fname.mbar*tearOff		0	$_fname(level)
option add *fname.mbar*indicatorOn	1	$_fname(level)
option add *fname.bbox.borderWidth	1	$_fname(level)
option add *fname.name.width		20	$_fname(level)
option add *fname.mbar.dir.width	15	$_fname(level)
option add *fname.max(recent)		10	$_fname(level)
option add *fname.list.width		20	$_fname(level)
option add *fname.list.height		8	$_fname(level)
option add *fname.list.selectBorderWidth 0	$_fname(level)
option add *fname.list.selectBackground white	$_fname(level)
option add *fname.list-frame.relief	groove	$_fname(level)
option add *fname.list-frame.borderWidth 3	$_fname(level)

option add *fname.button(font) \
  "*-helvetica-bold-r-normal--*-140-*" $_fname(level)
option add *fname.file(font) \
  "*-courier-bold-r-normal--*-140-*" $_fname(level)
option add *fname.nonmatch(font) \
  "*-courier-medium-r-normal--*-140-*" $_fname(level)
option add *fname.dir(font) \
  "*-courier-medium-r-normal--*-140-*" $_fname(level)
option add *fname.file(color)	  black		$_fname(level)
option add *fname.nonmatch(color) grey50	$_fname(level)
option add *fname.dir(color)	  black		$_fname(level)

set _fname(maxrecent)	  [option get .fname max(recent) *]
set _fname(file.font)	  [option get .fname file(font) *]
set _fname(button.font)	  [option get .fname button(font) *]

option add *fname.mbar*font   $_fname(file.font)	$_fname(level)
option add *fname.mbar.recent.font $_fname(button.font) $_fname(level)
option add *fname.list.font   $_fname(file.font)	$_fname(level)
option add *fname.Button.font $_fname(button.font)	$_fname(level)
option add *fname.Label.font  $_fname(button.font)	$_fname(level)
option add *fname*Menubutton.padX 9			$_fname(level)
option add *fname*Entry.font  $_fname(file.font)	$_fname(level)
option add *fname*Entry.selectBorderWidth 0		$_fname(level)
option add *fname*Entry.selectBackground white		$_fname(level)

#
#  The menu buttons
#
frame .fname.mbar -relief flat -bd 1; pack .fname.mbar -side top -fill x
menubutton .fname.mbar.dir -menu .fname.mbar.dir.menu \
  -textvariable _fname(dname)
menubutton .fname.mbar.recent -menu .fname.mbar.recent.menu \
  -text Recent -underline 0
menu .fname.mbar.dir.menu; menu .fname.mbar.recent.menu
pack .fname.mbar.dir -side left -fill both -expand true
pack .fname.mbar.recent -side right

#
#  The action buttons
#
frame .fname.bbox; pack .fname.bbox -side bottom -fill both
button .fname.cancel -text "Cancel"   -command _fname(Cancel)  -underline 0
button .fname.delete -text "Delete"   -command _fname(Delete)  -underline 0
button .fname.mkdir  -text "Make Dir" -command _fname(MakeDir) -underline 0
button .fname.filter -text "Filter"   -command _fname(Filter)  -underline 0
button .fname.ok     -text "OK"	      -command _fname(OK)      -underline 0 \
    -textvariable _fname(type)
pack .fname.cancel .fname.delete .fname.mkdir .fname.filter .fname.ok \
  -in .fname.bbox -side left -fill x -expand true

#
#  The title and file name area
#
frame .fname.box; pack .fname.box -side bottom -pady 2 -fill x
label .fname.title -textvariable _fname(title)
entry .fname.name -textvariable _fname(name)
pack .fname.title -in .fname.box -pady 1
pack .fname.name -in .fname.box -fill x -padx 5 -pady 1
bindtags .fname.name \
  {.fname.name _fname(map) Entry _fname(select.map) .fname all}

#
#  The file name list and scroll-bar
#
frame .fname.list-frame; frame .fname.list-box -relief flat -borderwidth 0
text .fname.list -yscrollcommand {.fname.list-scroll set} -padx 5 -pady 3 \
  -wrap word -exportselection false -insertontime 0 -cursor {}
scrollbar .fname.list-scroll -command {.fname.list yview}
pack .fname.list-scroll -in .fname.list-box -side right -fill y
pack .fname.list -in .fname.list-box -fill both -expand true
pack .fname.list-box -in .fname.list-frame -fill both -expand true \
    -padx 2 -pady 2
pack .fname.list-frame -in .fname -side top -fill both -expand true

#
#  Change the bindings so the name list works more like a listbox
#  (but we needed to use a Text widget since we wanted to be able to 
#  specify fonts and colors individually
#
bindtags .fname.list \
  {.fname.list _fname(list.map) _fname(map) _fname(select.map) .fname all}

#
#  The font/color combinations for matched files, nonmatched files,
#  and directory names in the list box
#
.fname.list tag configure file \
      -font [option get .fname file(font) *] \
      -foreground [option get .fname file(color) *]
.fname.list tag configure nonmatch \
      -font [option get .fname nonmatch(font) *] \
      -foreground [option get .fname nonmatch(color) *]
.fname.list tag configure dir \
      -font [option get .fname dir(font) *] \
      -foreground [option get .fname dir(color) *]


######################################################################
#
#  Make button short-cuts actually press the button
#
set _fname(invoke.groove) sunken
set _fname(invoke.raised) sunken
set _fname(invoke.ridge)  sunken
set _fname(invoke.sunken) raised

proc _fname(Invoke) {button} {
  global _fname errorCode errorInfo
  set relief [$button cget -relief]
  if [info exists _fname(invoke.$relief)] {
    $button configure -relief $_fname(invoke.$relief)
    update idletasks
  }
  set code [catch [list uplevel \#0 "$button invoke"] message]
  $button configure -relief $relief
  return -code $code -errorcode $errorCode -errorinfo $errorInfo $message
}

#
#  The button short-cuts
#
bind .fname <Alt-c> "_fname(Invoke) .fname.cancel"
bind .fname <Alt-d> "_fname(Invoke) .fname.delete"
bind .fname <Alt-m> "_fname(Invoke) .fname.mkdir"
bind .fname <Alt-f> "_fname(Invoke) .fname.filter"
bind .fname <Alt-o> "_fname(Invoke) .fname.ok"

bind .fname <Alt-s>    [bind .fname <Alt-o>]
bind .fname <Return>   [bind .fname <Alt-o>]
bind .fname <KP_Enter> [bind .fname <Alt-o>]
bind .fname <Escape>   [bind .fname <Alt-c>]


######################################################################
#
#  Get width of characters in directory button (hack)
#

#
#  Remove the indicator and any padding
#  Get the width of the button
#  Put back the indicator and padding
#  Get the difference of the widths
#  Get the character width
#
proc _fname(GetWidth) {} {
  global _fname
  set indicator [.fname.mbar.dir cget -indicatoron]
  set padx [.fname.mbar.dir cget -padx]
  .fname.mbar.dir configure -indicatoron 0 -padx 0
  set width [winfo reqwidth .fname.mbar.dir]
  .fname.mbar.dir configure -indicatoron $indicator -padx $padx
  set _fname(wextra) [expr [winfo reqwidth .fname.mbar.dir] - $width]
  set _fname(wchar) [expr $width / double([.fname.mbar.dir cget -width])]
}
_fname(GetWidth)

#
#  Make sure displayed name always fits into the button
#
bind .fname.mbar.dir <Configure> {_fname(SetDirName)}


######################################################################
#
#  The button actions
#

#
#  _fname(Close)
#
#  Close the name requester window
#
proc _fname(Close) {} {
  grab release .fname
  wm withdraw .fname
  update idletasks
}

#
#  _fname(Cancel) - cancel the name request
#
#  Just close the window when Cancel is pressed
#
proc _fname(Cancel) {} {_fname(Close)}

#
#  _fname(Delete) - delete a file
#
#  Make sure a file is specified
#  Get the file name
#  If it doesn't exist, tell the user
#  Make sure the user REALLY wants to delete it
#  Do /bin/rm or /bin/rmdir, depending on the type of file
#  Clear the name entry
#  Read the directory again
#
proc _fname(Delete) {} {
  global _fname
  _fname(CheckName) "You must select a file to delete"
  set name [_fname(Name)]
  if ![file exists $name] {_fname(Error) "File '$name' doesn't exist!"}
  if [_fname(Confirm) "Really delete '$name'?"] return
  if [file isdirectory $name] {
    if [catch "exec /bin/rmdir $name" message] {_fname(Error) $message}
  } else {
    if [catch "exec /bin/rm $name" message] {_fname(Error) $message}
  }
  set _fname(name) ""
  _fname(SetDirectory) $_fname(dir)
}

#
#  _fname(MakeDir) - make a new directory
#
#  Make sure a directory name is specified
#  Get the full name
#  Check to be sure the name doesn't include wierd characters
#  Check to make sure the directory doesn't already exist
#  Try to make the directiry
#  Clear the name entry
#  Move to the new directory
#
proc _fname(MakeDir) {} {
  global _fname
  _fname(CheckName) "You must provide a directory name"
  set dir [_fname(Name)]
  _fname(CheckSpecials) $dir "Create"
  if [file exists $dir] {_fname(Error) "'$dir' already exists!"}
  if [catch "exec /bin/mkdir $dir" message] {_fname(Error) $message}
  set _fname(name) ""
  _fname(SetDirectory) $dir
}


#
#  _fname(Filter) - change the filter
#
#  If there is a filter specified
#    Get the full name
#    Get the directory portion, and check that it exists
#    Set the filter to the name portion
#    If there is no name portion, default to "*"
#    Clear the name entry
#    Move to the specified directory
#  Otherwise
#    reload the current directory
#
proc _fname(Filter) {} {
  global _fname
  if {[string trim $_fname(name)] != ""} {
    set name [_fname(Name)]
    set dir [file dirname $name]
    if ![file isdirectory $dir] {_fname(Error) "No such directory '$dir'"}
    set _fname(filter) [file tail $name]
    if {$_fname(filter) == ""} {set _fname(filter) "*"}
    set _fname(name) ""
    _fname(SetDirectory) $dir
  } else {
    _fname(LoadDirectory)
  }
}

#
#  Shift-clicking the Filter button reloads the current directory
#
bind .fname.filter <Shift-ButtonRelease-1> {
  if {$tkPriv(window) == "%W"} {
    set tkPriv(buttonWindow) ""
    %W config -relief $tkPriv(relief)
    _fname(SetDirectory) $_fname(dir)
  }
  break;
}

#
#  Alt-clicking the Filter button toggles the showAll option
#
bind .fname.filter <Alt-ButtonRelease-1> {
  if {$tkPriv(window) == "%W"} {
    set tkPriv(buttonWindow) ""
    %W config -relief $tkPriv(relief)
    set _fname(showAll) [expr !$_fname(showAll)]
    _fname(SetDirectory) $_fname(dir)
  }
  break;
}

#
#  Control-clicking the Filter button shows the current filter
#
bind .fname.filter <Control-ButtonRelease-1> {
  if {$tkPriv(window) == "%W"} {
    set tkPriv(buttonWindow) ""
    %W config -relief $tkPriv(relief)
    set _fname(name) $_fname(filter)
  }
  break;
}


#
#  _fname(OK) - handle the Open or Save button
#
#  Make sure a name was entered
#  Get the full name
#  If it is a directory, move to that directory and don't end the request
#  For a SAVE event:
#    If the file already exists, ask the user if it should be overwritten
#    Make sure the name doesn't include wierd characters
#  For an OPEN even, make sure the file exists
#  Put the full name into the action script
#  Do the action
#  Close the window
#
proc _fname(OK) {} {
  global _fname
  _fname(CheckName) "You must supply a file name"
  set name [_fname(Name)]
  if [file isdirectory $name] {_fname(SetDirectory) $name; return}
  switch $_fname(type) {
    Save {
      if [file exists $name] \
	  {if [_fname(Confirm) "File '$name' already exists:  Overwrite it?"] \
	       return}
      _fname(CheckSpecials) $name "Save"
    }
    Open {if ![file exists $name] \
	      {_fname(Error) "File '$name' doesn't exists"}}
    default {}
  }
  if {$_fname(action) != ""} {
    regsub -all {%N} $_fname(action) $name action
    set _fname(action) {}
    uplevel \#0 $action
  }
  _fname(Close)
}

######################################################################
#
#  The main entry points to the file requester
#
proc _fname(NewFile) {args} {eval _fname(Request) Save $args}
proc _fname(OldFile) {args} {eval _fname(Request) Open $args}

#
#  _fname(Request) - put up the requester
#
#  iconify the requester if it is mapped (uniconifying brings it to the front)
#  Set the variables
#  Get the directory and name portions of the default name
#  Make sure the default name is not a directory
#  Show the currect directory
#  Set the default file name and search for it in the list
#  If there is a default name
#    Set the selection to include the name
#    Don't select the extension if one is given
#  Center the window if this is the first time it is open
#  Open the window
#  Update the "..." in the directory button, if needed
#  Focus on the entry
#
proc _fname(Request) {type title name action {ext ""} {dir ""}} {
  global _fname
  if [winfo ismapped .fname] {wm iconify .fname}
  set _fname(action) $action
  set _fname(title) $title
  set _fname(type) $type
  set _fname(filter) "*$ext"
  set dir [_fname(GetDirectory) $dir $_fname(dir)]
  set name [_fname(Normalize) $name]
  if {[string index $name 0] != "/"} {set name [_fname(Normalize) $dir/$name]}
  if {[file isdirectory $name]} {set dir $name; set name ""} \
      else {set dir [file dirname $name]; set name [file tail $name]}
  _fname(SetDirectory) $dir
  set _fname(name) $name; _fname(Search)
  if {$name != ""} {
    .fname.name selection range 0 end
    .fname.name icursor end
    if {$ext != "" && [file extension $name] == $ext} {
      set last [string length [file rootname $name]]
      .fname.name selection range 0 $last
      .fname.name icursor $last
    }
  }
  _fname(Center)
  wm deiconify .fname
  _fname(SetDirName)
  focus .fname.name
  if {[grab current .] != ""} {grab release [grab current .]}
  grab set .fname
}


######################################################################
#
#  Service routines that handle the file name list and directory lists
#


#
#  _fname(Center) - compute the location for the window
#
#  if the window isn't already centered (and possibly moved by the user)
#  Compute the center horizontally and 2/3 up vertically
#
proc _fname(Center) {} {
  global _fname
  if {! $_fname(centered)} {
    set x [expr [winfo screenwidth .fname]/2 - [winfo reqwidth .fname]/2 - \
		[winfo vrootx [winfo parent .fname]]]
    set y [expr [winfo screenheight .fname]/3 - [winfo reqheight .fname]/2 - \
		[winfo vrooty [winfo parent .fname]]]
    wm geom .fname +$x+$y
    set _fname(centered) 1
  }
}

#
#  _fname(CheckName) - make sure there is a name given
#
#  If there is no name in the entry, display an error message
#
proc _fname(CheckName) {message} {
  global _fname
  if {[string trim $_fname(name)] == ""} {_fname(Error) $message}
}

#
#  _fname(CheckSpecials) - check for wierd characters in a file name
#
#  If the name includes a funny character
#    Ask the user if he really wants to use the name
#      if not, return from the calling routine
#
proc _fname(CheckSpecials) {name action} {
  if [regexp {([\$&\*\(\)\|`\{\}\"'\?<>]|\[|\])} $name c] {
    if [_fname(Confirm) [concat "Name \"$name\" contains the special" \
	  "character \"$c\":  $action anyway?"]] {return -code return}
  }
}

#
#  _fname(Name) - get the full name of the spcified file
#
#  Get the name in the entry
#  Add the directory if needed
#  Handle ".." and "." constructions
#
proc _fname(Name) {} {
  global _fname
  set name [string trim $_fname(name)]
  if {[string index $name 0] != "/"} {set name "$_fname(dir)/$name"}
  return [_fname(Normalize) $name]
}


#
#  _fname(Normalize) - handle ".." and "." directory names
#
#  Get rid of spaces in the name
#  Get rid of multiple slashes
#  Remove redundent "./"
#  Remove "dir/../"
#  Remove trailing "/"
#
proc _fname(Normalize) {name} {
  regsub -all { } $name {} name
  while {[regsub -all {//} $name {/} name]} {}
  while {[regsub -all {(^|/)\./} $name {\1} name]} {}
  while {[regsub -all {/[^/]*/\.\.(/|$)} $name {\1} name]} {}
  regsub {/+$} $name {} name
  return $name
}

#
#  _fname(GetDirectory) - find a full directory from a name
#
#  Remove spaces from the directory
#  If the directory is relative
#    If there is no root, be relative to PWD otherwise relative to root
#  Return the full name
#
proc _fname(GetDirectory) {dir {root ""}} {
  global env
  set dir [string trim $dir]
  if {[string index $dir 0] != "/"} {
    if {$root == ""} {set dir $env(PWD)/$dir} else {set dir $root/$dir}
  }
  return [_fname(Normalize) $dir]
}

#
#  _fname(SetDirectory) - change the directory showing in the list
#
#  Get the full directory name
#  Remove the current parent directory menu
#  While there are more parent directories
#    Add the directory to the menu (it's action be to selects that directory)
#    Get the parent of the specified directory
#  Add the directory to the recent-directories menu
#  Load the specified directory
#  Clear the file name entry
#
proc _fname(SetDirectory) {dir} {
  global _fname 
  set _fname(dir) [_fname(GetDirectory) $dir $_fname(dir)]
  .fname.mbar.dir.menu delete 0 end
  while {$dir != "."} {
    if [file isdirectory $dir] {
      .fname.mbar.dir.menu add command -label $dir \
	  -command [list _fname(SetDirectory) $dir]
    } else {
      set _fname(dir) [file dirname $dir]
    }
    if {$dir == "/"} {set dir ""}
    set dir [file dirname $dir]
  }
  _fname(AddToRecent) $_fname(dir)
  _fname(LoadDirectory)
  set _fname(name) ""
  _fname(SetDirName)
}

#
#  _fname(SetDirName) - check for "..." in directory name button
#
#  Get the width of the name area (use reqwidth if necessary)
#  Find the number of characters that fit
#  If the directory name is too long
#    Get about half of them from the front and half from the back and
#      join them with "..."
#  Otherwise
#    Use the complete string
#
proc _fname(SetDirName) {} {
  global _fname
  set width [winfo width .fname.mbar.dir]
  if {$width == 1} {set width [winfo reqwidth .fname.mbar.dir]}
  set width [expr int(($width - $_fname(wextra)) / $_fname(wchar)) - 1]
  if {[string length $_fname(dir)] > $width && $width > 0} {
    set width [expr int($width/2)-1];
    set _fname(dname) [join [list \
	 [string range $_fname(dir) 0 $width] \
	 [string range $_fname(dir) \
           [expr [string length $_fname(dir)] - $width] end] \
       ] "..."]
  } else {
    set _fname(dname) $_fname(dir)
  }
}

#
#  _fname(AddToRecent) - add a directory to the recent-directory menu
#
#  Remove the directory if it is already on the list
#  Add the menu to the front of the list (don't hold too many, though)
#  Clear the current menu
#  Create a new menu from the list of recent names
#
proc _fname(AddToRecent) {dir} {
  global _fname
  regsub "(^| )${dir}( |\$)" $_fname(recent) {\1} _fname(recent)
  set _fname(recent) \
      [concat $dir [lrange $_fname(recent) 0 [expr $_fname(maxrecent) - 2]]]
  .fname.mbar.recent.menu delete 0 end
  foreach dir $_fname(recent) {
    .fname.mbar.recent.menu add command -label $dir \
	-command [list _fname(SetDirectory) $dir]
  }
}

#
#  _fname(LoadDirectory) - look up the file names in the current directory
#
#  Clear the file list and put up a message (in case this takes a while)
#  Make sure the message is showing
#  Clear the message
#  Get the matching file names, and get the first name in the list
#  Get the complete list of all files (including "Dot" files)
#  For each file in the complete list
#    If the file is a directory
#      If it is not "." or "..", add it to the file list as a directory
#    If it is the next matching file, add it into the list
#    If we are showing all and the file is not a dot-file, add it
#    If it is a match, remove it from the match list and get the next one
#  Remove the trailing \n
#  Reset the match pointer to the beginning of the list
#
proc _fname(LoadDirectory) {} {
  global _fname
  .fname.list delete 0.0 end
  .fname.list insert end "Loading..."
  update idletasks
  .fname.list delete 0.0 end
  set matches [lsort [glob -nocomplain $_fname(dir)/$_fname(filter)]]
  set next [lindex $matches 0]
  foreach file [lsort [glob -nocomplain $_fname(dir)/{.,}*]] {
    set name [file tail $file]
    if [file isdirectory $file] {
      if ![regexp {^\.\.?$} $name] {
	.fname.list insert end $name/\n dir
      }
    } elseif {$file == $next} {
      .fname.list insert end $name\n file
    } elseif {$_fname(showAll) && [string index $name 0] != "."} {
      .fname.list insert end $name\n nonmatch
    }
    if {$file == $next} {
      set matches [lrange $matches 1 end]
      set next [lindex $matches 0]
    }
  }
  .fname.list delete {end -2 c} end
  .fname.list mark set match 1.0
}

######################################################################
#
#  Keymapings to make the file list work like a listbox
#


#
#  Make picking and dragging work
#
bind _fname(list.map) <1>		{_fname(Select) @%x,%y Pick; focus %W}
bind _fname(list.map) <B1-Motion>	{_fname(Select) @%x,%y Pick}
bind _fname(list.map) <B1-Leave>	{_fname(AutoScan) %W %y}
bind _fname(list.map) <B1-Enter>	_fname(CancelRepeat)
bind _fname(list.map) <ButtonRelease-1> _fname(CancelRepeat)
bind _fname(list.map) <Double-1> \
    {_fname(Select) @%x,%y Pick; _fname(OK)}

#
#  Make scrolling work
#
bind _fname(list.map) <2>		[bind Text <2>]
bind _fname(list.map) <B2-Motion>	[bind Text <B2-Motion>]

#
#  Left-arrow moves to the parent directory
#  Riight arrow selects the current file
#  Backspace and Delete do a backspace in the Entry
#
bind _fname(list.map) <Left>		{.fname.mbar.dir.menu invoke 1}
bind _fname(list.map) <Right>		{_fname(OK)}
bind _fname(list.map) <BackSpace>	{tkEntryBackspace .fname.name}
bind _fname(list.map) <Delete>		[bind _fname(map) <BackSpace>]

#
#  If there is no selection or the selection is not a complete line
#    Try to complete a file name
#    Otherwise (if no completion) switch between the list and the entry
#  Control-Tab cycles through all the widgets
#
bind _fname(map) <Tab> {
  if {[.fname.list tag range sel] != {} &&
      [.fname.list compare {sel.first + 1 line} != sel.last]} _fname(Complete)
  if {"%W" == ".fname.name"} {focus .fname.list} else {focus .fname.name}
  break
}
bind _fname(map) <Control-Tab> {
  if {"%W" == ".fname.name"} {focus .fname.list} else {focus .fname.name}
}
bind _fname(map) <Shift-Tab>	    { }

#
#  Space also does a file-name completion
#  Control-space adds a space (if you REALLY want one)
#
bind _fname(map) <space>	     {_fname(Complete); break}
bind _fname(map) <Control-space>     { }

#
#  Shift-return selects the parent directory
#  Up or Control-P select the previous file in the list
#  Down or Control-N select the previous file in the list
#  Select presses the OK button
#  Next or Shift-Up or ESC-v scrolls forward a page
#  Prior or Shift-Down or Control-V scrolls backward a page
#  Home or ESC-< or Control-Up moves to the start of the list
#  End or ESC-> or Control-Down moves to the end of the list
#  Control-\ or Control-U removes any selection and clears the name entry
#  
bind _fname(map) <Shift-Return>	    {.fname.mbar.dir.menu invoke 1; break}
bind _fname(map) <Next> \
    {_fname(Select) [tkTextScrollPages .fname.list 1] Forward; break}
bind _fname(map) <Prior> \
    {_fname(Select) [tkTextScrollPages .fname.list -1] Backward; break}
bind _fname(map) <Up>		    {_fname(Select) {match - 1 line} Backward}
bind _fname(map) <Down>		    {_fname(Select) {match + 1 line} Forward}
bind _fname(map) <Select>	    [bind .fname <Alt-o>]
bind _fname(map) <Home>		    {_fname(Select) 1.0 Forward; break}
bind _fname(map) <End>		    {_fname(Select) end Backward; break}
bind _fname(map) <Control-backslash> \
    {.fname.list tag remove sel 1.0 end; set _fname(name) ""; break}
bind _fname(map) <Control-p>	    [bind _fname(map) <Up>]
bind _fname(map) <Control-n>	    [bind _fname(map) <Down>]
bind _fname(map) <Escape>	    {break}
bind _fname(map) <Escape><less>	    [bind _fname(map) <Home>]
bind _fname(map) <Escape><greater>  [bind _fname(map) <End>]
bind _fname(map) <Escape>v	    [bind _fname(map) <Prior>]
bind _fname(map) <Control-v>	    [bind _fname(map) <Next>]
bind _fname(map) <Shift-Up>	    [bind _fname(map) <Prior>]
bind _fname(map) <Shift-Down>	    [bind _fname(map) <Next>]
bind _fname(map) <Control-Up>	    [bind _fname(map) <Home>]
bind _fname(map) <Control-Down>	    [bind _fname(map) <End>]
bind _fname(map) <Control-u>	    [bind _fname(map) <Control-backslash>]

#
#  For any keypress:
#
#  If we are in the file list
#    If the character is a printable one, add it to the file name
#    Search for the next match
#  Otherwise (in the name entry)
#    If we are automatically selecting in the file list, search for the name
#
bind _fname(select.map) <KeyPress> {
  if {"%W" == ".fname.list"} {
    if {"%A" >= " " && "%A" < "\x7f"} {tkEntryInsert .fname.name %A}
    _fname(Search)
  } else {
    if {$_fname(autoSelect)} _fname(Search)
  }
}

#
#  _fname(Select) - select an entry in the file list
#
#  Set the name entry to nothing
#  Remove any selection
#  Set the math to the start of the next available line (skip nonmatches)
#  If we are at the end, back up a line
#  If we are not picking by mouse clicks, or if the line can be selected
#    Select the matched line
#    Set the name entry to contain the selected line
#  Make sure the list shows the selected item
#
proc _fname(Select) {i {direction Forward}} {
  global _fname 
  set _fname(name) ""
  .fname.list tag remove sel 1.0 end
  .fname.list mark set match [_fname(Adjust$direction) "$i linestart"]
  if [.fname.list compare match == end] \
      {.fname.list mark set match {end - 1 line}}
  if {$direction != "Pick" || [.fname.list tag names match] != "nonmatch"} {
    .fname.list tag add sel match {match + 1 line}
    set _fname(name) [.fname.list get match {match lineend}]
  }
  .fname.list see match
}

#
#  _fname(AutoScan) - handle dragging off the top or bottom of the list
#
#  If the mouse is below the list, scoll down, otherwise
#  If the mouse is above the list, scroll up, otherwise return
#  Select the new item that has scrolled on
#  Do it again after the desired amount of time
#
proc _fname(AutoScan) {w y} {
  global _fname
  if {$y >= [winfo height $w]} {$w yview scroll 1 unit} \
      elseif {$y < 0} {$w yview scroll -1 unit} else {return}
  _fname(Select) @0,$y Pick
  set _fname(afterID) [after $_fname(scanRate) _fname(AutoScan) $w $y]
}

#
#  _fname(CancelRepeat) - remove a pending AutoScan command
#
#  If there is a pending AutoScan (dragging off the list)
#    Cancel it
#    Clear the indicator
#
proc _fname(CancelRepeat) {} {
  global _fname
  if {$_fname(afterID) != ""} {
    after cancel $_fname(afterID)
    set _fname(afterID) {}
  }
}


#
#  _fname(NextForward) - look forward for the next file that is a match
#
#  While we are not at the end of the file list
#    Move ahead a line
#    If the file can be selected, return its position
#  Return "end"
#
proc _fname(NextForward) {{i match}} {
  while {[.fname.list compare $i < {end - 1 line}]} {
    set i [.fname.list index "$i + 1 line"]
    if {[.fname.list tag names $i] != "nonmatch"} {return $i}
  }
  return end
}

#
#  _fname(NextBackward) - look backward for the previous file that is a match
#
#  If we are already at the beginning, return 0.0 (before the beginning)
#  While we are note at the beginning of the file list
#    Move back one line
#    If the line matches, return the position
#  If the first line matches, return than
#  Otherwise return "0.0" (no match)
#
proc _fname(NextBackward) {{i match}} {
  if {[.fname.list compare $i == 1.0]} {return 0.0}
  while {[.fname.list compare $i > 1.0]} {
    set i [.fname.list index "$i - 1 line"]
    if {[.fname.list tag names $i] != "nonmatch"} {return $i}
  }
  if {[.fname.list tag names 1.0] != "nonmatch"} {return 1.0}
  return 0.0
}

#
#  _fname(AdjustForward) - skip forward over non-matched files
#
#  If the selected file is not a match
#    Move forward to the next match
#    If we are at the end
#      Move backward to the previous match
#      If non (i.e., no matched files), go to the end
#  Return the matched line
#
proc _fname(AdjustForward) {{i match}} {
  if {[.fname.list tag names $i] == "nonmatch"} {
    set i [_fname(NextForward) $i]
    if {$i == "end"} {
      set i [_fname(NextBackward) end]
      if {$i == "0.0"} {set i end}
    }
  }
  return $i
}

#
#  _fname(AdjustBackward) - skip backward over non-matched files
#
#  If the selected file is not a match
#    Skip backward to the previous match
#    If non, skip forward to the next one
#  Return the matched line
#
proc _fname(AdjustBackward) {{i match}} {
  if {[.fname.list tag names $i] == "nonmatch"} {
    set i [_fname(NextBackward) $i]
    if {$i == "0.0"} {set i [_fname(NextForward) 1.0]}
  }
  return $i
}

#
#  _fname(AdjustPick) - return the selected line
#
#  return the current line
#
proc _fname(AdjustPick) {i} {return [.fname.list index "$i linestart"]}


#
#  _fname(SearchForward) - look forward for a file prefix
#
#  While we're not at the end of the list
#    If the current file has has the correct file prefix, we're done
#    Otherwise search forward for the next selectable file
#  Return the located line
#
proc _fname(SearchForward) {string {i match}} {
  set len [string length $string]
  while {$i != "end"} {
    set test [.fname.list get $i "$i + $len chars"]
    regsub {/} $test {} test
    if {[string compare $test $string] >= 0} break
    set i [_fname(NextForward) $i]
  }
  return $i
}

#
#  _fname(SearchBackward) - look backward for a file prefix
#
#  While we're not at the beginning of the list
#    If the current file as the right prefix, we're done
#    Otherwise back up to the previous selectable file
#  Return the located line
#
proc _fname(SearchBackward) {string {i match}} {
  set len [string length $string]
  while {$i != "0.0"} {
    set test [.fname.list get $i "$i + $len chars"]
    regsub {/} $test {} test
    if {[string compare $test $string] <= 0 && $test != ""} break
    set i [_fname(NextBackward) $i]
  }
  return $i
}

#
#  _fname(SkipForward) - skip forward over lines with a matching prefix
#
#  Get the next selectable line
#  While we're not at the end of th list
#    If the file has a different prefix, we're done
#    Otherwise record this line and go on to the next
#  Return the last matching line
#
proc _fname(SkipForward) {string {i match}} {
  set len [string length $string]
  set j [_fname(NextForward) $i]
  while {$j != "end"} {
    if [string compare $string [.fname.list get $j "$j + $len chars"]] break
    set i $j
    set j [_fname(NextForward) $i]
  }
  return $i
}

#
#  _fname(SkipBackward) - skip backward over lines with a matching prefix
#
#  Get the previous selectable line
#  While we're not at the beginning of the list
#    If the file doesn't have a matching prefix, we're done
#    Otherwise record this line and go to the previous one
#  Return the last matching line
#
proc _fname(SkipBackward) {string {i match}} {
  set len [string length $string]
  set j [_fname(NextBackward) $i]
  while {$j != "0.0"} {
    if [string compare $string [.fname.list get $j "$j + $len chars"]] break
    set i $j
    set j [_fname(NextBackward) $i]
  }
  return $i
}

#
#  _fname(Search) - look for a file matching the current name entry
#
#  Remove any selection in the file list
#  If there is a name to look for
#    Get the current match prefix
#    If the prefix is after the current name, search backward from the match
#    Otherwise search forward from the most recent match
#    Get the prefix from the matched line
#    If the prefix matches the typed name
#      Backup past any others that match
#      Select the matching prefix in the list
#      If the whole line is matched and there are no additional matches
#	 Select the rest of the line
#    Make sure the selection is showing
#
proc _fname(Search) {} {
  global _fname
  .fname.list tag remove sel 1.0 end
  if {$_fname(name) != ""} {
    set len [string length $_fname(name)]
    set name [.fname.list get match "match + $len chars"]
    regsub -all {/} $name {} name
    if {[string compare $name $_fname(name)] == 1 || $name == ""} {
      .fname.list mark set match [_fname(SearchBackward) $_fname(name)]
    } else {
      .fname.list mark set match [_fname(SearchForward) $_fname(name)]
    }
    set name [.fname.list get match "match + $len chars"]
    if {[string compare $name $_fname(name)] == 0} {
      .fname.list mark set match [_fname(SkipBackward) $_fname(name)]
      .fname.list tag add sel match "match + $len chars"
      if {[.fname.list compare sel.last == {match lineend}] &&
	  [.fname.list compare match == [_fname(SkipForward) $_fname(name)]]} \
	      {.fname.list tag add sel match {match + 1 line}}
    }
    .fname.list see match
  }
}


#
#  _fname(Complete) - try to complete a file name
#
#  If there is no name to complete, return
#  If there is a selection that is not already the full line
#    Look for additional lines with the same prefix as the current line
#    If there are none, select the current line, otherwise
#      Get the number of matching lines
#      Get the complete selected line
#      For each line matching the current prefix
#	 Look for the longest prefix match with the previous lines
#	 Trim the prefix to this length (all previous line match this far)
#      If there is no extension possible, return, otherwise
#      Set the name entry to the longest prefix match
#      Select the matched prefix
#
#  Do not process this key further (break the bindings)
#
proc _fname(Complete) {} {
  global _fname
  if {[set len [string length $_fname(name)]] == 0} return
  if {[.fname.list tag range sel] != {} &&
      [.fname.list compare sel.last != {sel.first + 1 line}]} {
    set i [_fname(SkipForward) $_fname(name)]
    if [.fname.list compare $i == match] {_fname(Select) match} else {
      set i [expr int([.fname.list index $i] - [.fname.list index match])]
      set name [.fname.list get match {match lineend}]
      for {set j 1} {$j < $i} {incr j} {
	set l [string length $name]
	for {set k $len} {$k <= $l} {incr k} {
	  if {[string index $name $k] !=
	      [.fname.list get "match+$j lines+$k chars"]} break
	}
	set name [string range $name 0 [expr $k-1]]  
      }
      if {[string length $name] == $len} return
      set _fname(name) $name
      .fname.list tag remove sel 1.0 end
      .fname.list tag add sel match "match + [string length $name] chars"
    }
  }
  return -code break
}



######################################################################
#
#  Error and confirmation messages
#


#
#  _fname(Error) - process an error message
#
#  Return an error condition and message
#
proc _fname(Error) {message} \
  {return -code error -errorcode _FNAME_ERROR $message}

#
#  Defaults for the confirmation window
#
option add *confirm.msg.font \
  "*-helvetica-medium-r-normal--*-180-*" $_fname(level)
option add *confirm.msg.wrapLength 4i	$_fname(level)

#
#  Remove font specification from tk_dialog so we can 
#  supply our own (via the option command)
#
auto_load tk_dialog
set _fname(dialog) [info body tk_dialog]
regsub "label \\\$w.msg \[^\n\]*" $_fname(dialog) \
       {label $w.msg -text $text} _fname(dialog)
proc _fname(Dialog) [info args tk_dialog] $_fname(dialog)
unset _fname(dialog)


#
#  _fname(Confirm) - ask a "Yes or No" question and wait for a reply
#
#  As the question and wait for a reply
#  Make sure the screen updates after the request is removed
#  Return the result
#
proc _fname(Confirm) {question {default 1}} {
  global _program
  set result [_fname(Dialog) .confirm "$_program(id):  Confirmation" \
		  $question {} $default Yes No]
  update idletasks
  return $result
}
