Require vector.tcl
Require complex.tcl

if [info exists debug] {
  proc _test {args} {
    set line \
      [string trim [uplevel [list subst -nobackslashes -novariables $args]]]
    set list [uplevel _expr(Tokenize) $line]
    set tree [uplevel [list _expr(Parse) $list]]
    return [uplevel [list _expr(Build) $tree]]
  }
}

proc _expr(Clear) {} {global _parse; catch {unset _parse}}

proc _expr(Expr) {args} {
  global _parse _expr
  set line \
    [string trim [uplevel [list subst -novariables -nobackslashes $args]]]
  if [info exists _parse($line)] {return [uplevel $_parse($line)]}
  if [regexp -- {^-?[0-9]*(\.[0-9]*)?([eE]-?[0-9]+)?$} $line] {return $line}
  regsub -all "(^|\[^a-z\])$_expr(builtin)\\(" $line {} newline
  if ![regexp {[a-zA-Z^\{\}\,]} $newline] {return [uplevel expr $line]}
  set list [uplevel _expr(Tokenize) $line]
  set list [uplevel [list _expr(Parse) $list]]
  set list [uplevel [list _expr(Build) $list]]
  set _parse($line) $list
  return [uplevel [list eval $list]]
}

proc _expr(Build) {tree} {
  return [lrange [_expr(BuildExpr) $tree] 1 end]
}
proc _expr(BuildExpr) {tree} {
  set data [lindex $tree 1]
  switch [lindex $tree 0] {
    var {return [list [expr {[llength [uplevel 2 set $data]] > 1}] set $data]}
    num {return [list 0 list $data]}
    op {_expr(BuildOp)}
    fn {_expr(BuildFn)}
    list {_expr(BuildList)}
    str {return "0 expr \{\"$data\"\}"}
  }
}

proc _expr(BuildFn) {} {
  global _expr
  upvar tree tree
  set name [lindex $tree 1]
  set expr [uplevel 2 [list _expr(BuildExpr) [lindex $tree 2]]]
  set data [lindex $expr 2]
  set list [lindex $expr 0]
  set const [expr {[lindex $expr 1] == "list"}]
  set fn $_expr(fn:$name)
  if {[lindex $fn 1] != ""} {set name [lindex $fn 1]}
  if {[lindex $fn 0] && (!$list || [lindex $fn 2] == "")} {
    switch [lindex $expr 1] {
      expr {set expr $data}
      eval {
        if {$list} {
          set i 1; set len [llength $data]
          set expr {}
          while {$i < $len} {
            for {set j $i} {![info complete [join [lrange $data $i $j]]]} \
              {incr j} {}
            set e [join [lrange $data $i $j]]
            if {[lindex $data $i] == "\[expr"} \
              {set e [string range $e 6 [expr [string length $e] - 2]]}
            lappend expr $e
            set i [expr $j+1]
          }
          set expr [join $expr ,]
        } else {set expr "\[$data\]"}
      }
      list {set expr [join [lrange $expr 2 end] ,]}
      set {set expr "\$$data"}
    }
    set expr "${name}($expr)"
    if {$const} {return [list 0 list [expr $expr]]}
    return [list 0 expr $expr]
  } else {
    if {$list} {
      switch -- [lindex $fn 2] {
        0 {set list 0}
        {} {return -code error \
             "Function '$name' does not take vector arguments"}
      }
      if {[lindex $fn 3] != ""} {set name [lindex $fn 3]}
    }
    switch [lindex $expr 1] {
      expr {set expr "\[expr $data]"}
      list {set expr [lrange $expr 2 end]}
      eval {set expr "\[$data\]"}
      set {set expr "\$$data"}
    }
    set expr [list $name $expr]
    if {$const} {return [list $list list [eval $expr]]}
    return [list $list eval [join $expr]]
  }
}

proc _expr(BuildList) {} {
  upvar tree tree
  set list {list}
  set const 1
  foreach expr [lrange $tree 1 end] {
    set expr [uplevel 2 [list _expr(BuildExpr) $expr]]
    set data [lindex $expr 2]
    switch [lindex $expr 1] {
      expr {lappend list "\[expr $data\]"; set const 0}
      eval {lappend list "\[$data\]"; set const 0}
      list {lappend list [list [lrange $expr 2 end]]}
      set {lappend list \$$data; set const 0}
    }
  }
  if {$const} {return "1 [join $list]"}
  return [list 1 eval  [join $list { }]]
}

proc _expr(BuildOp) {} {
  global _expr
  upvar tree tree
  set op $_expr(op:[lindex $tree 1])
  set prec [lindex $op 0]
  set lop [lindex $tree 2]; set lexpr [uplevel 2 [list _expr(BuildExpr) $lop]]
  set lconst [expr {[lindex $lexpr 1] == "list"}]
  set llist [lindex $lexpr 0]
  switch [lindex $lexpr 1] {
    eval {set lexpr "\[[lindex $lexpr 2]\]"}
    expr {set lexpr [lindex $lexpr 2]}
    list {set lexpr [list [lrange $lexpr 2 end]]}
    set {set lexpr "\$[lindex $lexpr 2]"}
  }
  set rop [lindex $tree 3]
  if {$rop == ""} {
    if {$llist} {
      set expr [list [lindex $op 3] $lexpr]
      switch -- [lindex $op 5] {
        0 {set llist 0}
        {} {return -code error \
             "Operator '[lindex $op 3]' does not take a vector argument"}
      }
      if {$lconst} {return [list $llist list [eval $expr]]}
      return [list $llist eval [join $expr]]
    }
    if {[lindex $op 2]} {set expr [list 0 expr "[lindex $op 3]($lexpr)"]} \
      else {set expr [list 0 eval [list [lindex $op 3] $lexpr]]}
    if {$lconst} {set expr [list 0 list [eval [lrange $expr 1 end]]]}
    return $expr
  }
  set rexpr [uplevel 2 [list _expr(BuildExpr) $rop]]
  set rconst [expr {[lindex $rexpr 1] == "list"}]
  set rlist [lindex $rexpr 0]
  switch [lindex $rexpr 1] {
    eval {set rexpr "\[[lindex $rexpr 2]\]"}
    expr {set rexpr [lindex $rexpr 2]}
    list {set rexpr [list [lrange $rexpr 2 end]]}
    set {set rexpr "\$[lindex $rexpr 2]"}
  }
  if {![lindex $op 2] || $llist || $rlist} {
    set list 0
    if {$llist || $rlist} {
      switch -- [lindex $op 5] {
        1 {set list 1}
       -1 {set list $rlist}
        {} {return -code error \
             "Operator '[lindex $op 3]' does not take vector arguments"}
      }
    }
    if {[lindex $op 6] == ""} \
      {set expr [list $list eval "[lindex $op 3] $lexpr $rexpr"]} \
     else \
      {set expr [_expr(Build[lindex $op 6]) $llist $lexpr $rlist $rexpr]}
    if {!$lconst || !$rconst} {return $expr}
    return [join [list $list list [eval [lrange $expr 1 end]]]]
  }
  if {[lindex $lop 0] == "op" && [string index $lexpr 0] != "\["} {
    set lop $_expr(op:[lindex $lop 1])
    set lprec [lindex $lop 0]
    if {$lprec < $prec || ($lprec == $prec && [lindex $lop 1] == "right")} \
      {set lexpr "($lexpr)"}
  }
  if {[lindex $op 3] == "/"} {set rexpr "double($rexpr)"} else {
    if {[lindex $rop 0] == "op" && [string index $rexpr 0] != "\["} {
      set rop $_expr(op:[lindex $rop 1])
      set rprec [lindex $rop 0]
      if {$rprec < $prec || ($rprec == $prec && [lindex $rop 1] == "left")} \
	{set rexpr "($rexpr)"}
    }
    if {[string index $rexpr 0] == "-"} {set rexpr "($rexpr)"}
  }
  set expr [list 0 expr "$lexpr[lindex $op 3]$rexpr"]
  if {$lconst && $rconst} {set expr [list 0 list [eval [lrange $expr 1 end]]]}
  return $expr
}

proc _expr(Parse) {tokens} {
  global _expr
  set stack {{open start}}
  set apply {*}; set ptype {paren}

  foreach token $tokens {
    set state [lindex [lindex $stack end] 0]
    switch [lindex $token 0] {
      op {
        set op $_expr(op:[lindex $token 1])
        set type [lindex $op 4]
        if {$state == "expr"} {
          if {$type == "both" && $apply == ","} {
            _expr(PushOp) $apply
            lappend stack $token
          } else {
            if {$type == "unary"} {_expr(PushOp) $apply}
            _expr(PushOp) [lindex $token 1]
          }
        } else {
          if {$type == "bin"} \
            {return -code error "Missing operand before '[lindex $token 1]'"}
          lappend stack $token
        }
      }
      var - num - str {
        if {$state == "expr"} {_expr(PushOp) $apply}
        lappend stack [list "expr" $token]
      }
      fn {
        if {$state == "expr"} {_expr(PushOp) $apply}
        lappend stack $token
      }
      open {
        if {$state == "expr"} {_expr(PushOp) $apply}
        lappend stack $token
	lappend pstack $ptype
        set ptype [lindex $token 1]
      }
      close {
        _expr(Close) [lindex $token 1]
        set ptype [lindex $pstack end]
	set pstack [lreplace $pstack end end]
      }
      space {}
      default {return -code error \
                 "Parse return -code error: unknown token '$token'"}
    }
    if {$token == "space"} {
      if {$ptype == "brace"} {set apply ","}  else {set apply "space"}
    } else {set apply {*}}
  }
  _expr(Close) start
  return [lindex [lindex $stack 0] 1]
}

proc _expr(PushOp) {oname} {
  global _expr
  upvar stack stack
  set op [list op $oname]
  while 1 {
    set prev [lindex $stack [expr [llength $stack] - 2]]
    set pname [lindex $prev 0]
    if {$pname == "op"} {set pname [lindex $prev 1]}
    set pprec [lindex $_expr(op:$pname) 0]
    set oprec [lindex $_expr(op:$oname) 0]
    if {$pprec > $oprec || ($pprec == $oprec && 
          [lindex $_expr(op:$oname) 1] == "left")} {
      if {$pname == "space"} {set prev [lreplace $prev 1 1 {*}]}
      lappend prev [lindex [lindex $stack end] 1]
      set stack [lreplace [lreplace $stack end end] end end \
         [list "expr" $prev]]
    } elseif {$oname == ","} {
      if {[lindex $prev 1] == "start"} \
        {return -code error "Commas can't appear outside of delimiters"}
      lappend prev [lindex [lindex $stack end] 1]
      set stack [lreplace [lreplace $stack end end] end end $prev]
      return
    } else {
      lappend op [lindex [lindex $stack end] 1]
      set stack [lreplace $stack end end $op]
      return
    }
  }
}

proc _expr(Close) {type} {
  upvar stack stack
  while 1 {
    set top [lindex $stack end]
    switch [lindex $top 0] {
      expr {
        set n [expr [llength $stack] - 2]
        set prev [lindex $stack $n]
        lappend prev [lindex $top 1]
        if {[lindex $prev 0] == "fn" || [lindex $prev 0] == "op"} {
          if {[lrange $prev 0 1] == "op space"} \
            {set prev [lreplace $prev 1 1 {*}]}
          set prev [list "expr" $prev]
        }
        set stack [lreplace $stack $n end $prev]
        set top [lindex $stack end]
      }
      open {
        if {[lindex $top 1] == $type} {
          if {[llength $top] == 2} {return -code error "Missing expression"}
          if {[llength $top] == 3} {set top [lindex $top 2]} \
            else {set top [lreplace $top 0 1 "list"]}
          set prev [lindex $stack [expr [llength $stack] - 2]]
          if {[lindex $prev 0] == "fn"} {
            lappend prev $top
	    set top $prev
	    set stack [lreplace $stack end end]
          }
          set stack [lreplace $stack end end [list "expr" $top]]
          return
        }
        if {[lindex $top 1] == "start"} \
          {return -code error "Extra close-$type"}
        if {$type == "start"} \
          {return -code error "Missing close-[lindex $top 1]"}
        return -code error "Mismatched delimiters"
      }
      op {return -code error "Missing operand for '[lindex $top 1]'"}
      fn {return -code error "Missing parameter for '[lindex $top 1]'"}
    }
  }
}

proc _expr(Tokenize) {args} {
  global _expr _const
  set line [subst -nocommands -novariables $args]
  set list {}
  set len [string length $line]; set i 0
  while {$i < $len} {
    switch -glob -- [string index $line $i] {
      {[a-zA-Z_]} {
        regexp {[a-zA-Z_0-9]+} [string range $line $i end] name
        incr i [string length $name]
        if {[info exists _expr(fn:$name)]} {
          lappend list [list "fn" $name]
        } else {
          if {![uplevel [list info exists $name]]} \
            {return -code error "Unknown variable '$name'"}
          if {[info exists _const($name)]} \
            {lappend list [list "num" $_const($name)]} \
           else \
            {lappend list [list "var" $name]}
        }
      }
      {[0-9.]} {
        regexp -- {-?[0-9]*(\.[0-9]*)?([eE]-?[0-9]+)?} \
             [string range $line $i end] number
        incr i [string length $number]
        if {$number == "-"} {lappend list {op -}} \
          elseif {$number == "."} {lappend list {op .}} \
          else {lappend list [list "num" $number]}
      }
      {(} {lappend list {open paren}; incr i}
      {)} {lappend list {close paren}; incr i}
      "{" {lappend list {open brace}; incr i}
      "}" {lappend list {close brace}; incr i}
      { } - "\n" - "\t" {
        regexp "^\[ \n\t\]+" [string range $line $i end] spaces
        incr i [string length $spaces]
        lappend list {space}
      }
      {$} {
        if ![regexp {^\$({[^\}]+}|[a-zA-Z_0-9]+(\([^\)]*\))?|.)} \
                [string range $line $i end] var] {set var "$"}
        incr i [string length $var]
        set var [string range $var 1 end]
        if {[string index $var 0] == "\{"} \
          {set var [string range $var 1 [expr [string length $var] - 2]]}
        if {$var == ""} \
          {return -code error "Bad variable name in expression '$line'"}
        if {![uplevel [list info exists $var]]} \
          {return -code error "Unknown variable '$var'"}
        if {$var == "pi"} {lappend list [list "num" [uplevel set $var]]} \
          else {lappend list [list "var" $var]}
      }
      {"} {
        if ![regexp {^"([^"\\]|\\.)*"} [string range $line $i end] string] \
          {return -code error "Missing close-quote"}
        incr i [string length $string]
        lappend list [list "str" [string range $string 1 \
           [expr [string length $string] - 2]]]
      }
      default {
        set op [string range $line $i [expr $i+2]]
        while {![info exists _expr(op:$op)] &&
                [string length $op] > 1} \
          {set op [string range $op 0 [expr [string length $op] - 2]]}
        incr i [string length $op]
        if ![info exists _expr(op:$op)] \
          {return -code error "Unknown operator '$op'"}
        lappend list [list "op" $op]
      }
    }
  }
  return $list
}

proc Constant {name value} {global _const; set _const($name) $value}

Constant pi $pi


set _expr(op:!)   {16 right 1 ! unary}
set _expr(op:~)   {16 right 1 ~ unary}

set _expr(op:@)   {15 left 0 Extract bin -1}

set _expr(op:^)   {14 right 0 ^ bin 1 Power}

set _expr(op:*)   {13 left 1 *  bin 1}
set _expr(op:/)   {13 left 1 /  bin 1}
set _expr(op:%)   {13 left 1 %  bin}
set _expr(op:><)  {13 left 0 >< bin 1}
set _expr(op:.)   {13 left 0 Dot bin 0}

set _expr(op:fn)  {12 right 0 {} unary}

set _expr(op:space) {11 left 1 * bin 1}

set _expr(op:+)   {10 left 1 + both 1}
set _expr(op:-)   {10 left 1 - both 1}

set _expr(op:<<)  {9 left 1 << bin 1}
set _expr(op:>>)  {9 left 1 >> bin 1}

set _expr(op:<)   {8 left 1 <  bin}
set _expr(op:>)   {8 left 1 >  bin}
set _expr(op:<=)  {8 left 1 <= bin}
set _expr(op:>=)  {8 left 1 >= bin}

set _expr(op:==)  {7 left 1 == bin 0}
set _expr(op:!=)  {7 left 1 != bin 0}
set _expr(op:===) {7 left 0 === bin 0}
set _expr(op:!==) {7 left 0 !== bin 0}

set _expr(op:&)   {6 left 1 &  bin}
set _expr(op:^^)  {5 left 1 ^  bin}
set _expr(op:|)   {4 left 1 |  bin}
set _expr(op:&&)  {3 left 1 && bin}
set _expr(op:||)  {2 left 1 || bin}

set _expr(op:,)   {1 left 0 {} bin}

set _expr(op:open)  {0}


set _expr(fn:cos)   {1 {} 1 Cos}
set _expr(fn:sin)   {1 {} 1 Sin}
set _expr(fn:tan)   {1 {} 1 Tan}
set _expr(fn:acos)  1
set _expr(fn:asin)  1
set _expr(fn:atan)  1
set _expr(fn:atan2) 1
set _expr(fn:cosh)  1
set _expr(fn:sinh)  1
set _expr(fn:tanh)  1
set _expr(fn:hypot) 1
set _expr(fn:exp)   {1 {} 1 Exp}
set _expr(fn:log)   {1 {} 1 Log}
set _expr(fn:log10) 1
set _expr(fn:ln)    {1 log 1 Log}
set _expr(fn:pow)   1
set _expr(fn:sqrt)  {1 {} 1 Sqrt}
set _expr(fn:abs)   {1 {} 0 Norm}
set _expr(fn:ceil)  1
set _expr(fn:floor) 1
set _expr(fn:round) 1
set _expr(fn:int)   1
set _expr(fn:double) 1
set _expr(fn:fmod)  1

set _expr(builtin) [join [list \
  {a?(sin|cos|tan)h?} atan2 hypot exp {log(10)?} pow sqrt \
  abs ceil floor round int doubel fmod \
] {|}]


set _expr(fn:Sqrt)  {1 sqrt 1 Sqrt}
set _expr(fn:Sgn)   {0 Sgn 0}
set _expr(fn:Norm)  {0 Norm 0}
set _expr(fn:Unit)  {0 Unit 1}
set _expr(fn:Conj)  {0 Conj 1}
set _expr(fn:Arg)   {0 Arg 0}
set _expr(fn:Mod)   {0 Mod 0}
set _expr(fn:Re)    {0 Re 0}
set _expr(fn:Im)    {0 Im 0}

proc _expr(BuildPower) {llist lexpr rlist rexpr} {
  if {$llist || $rlist} {return [list 1 eval "^ $lexpr $rexpr"]}
  return [list 0 expr "pow($lexpr,$rexpr)"]
}



set _expr(useOldLet) 0
if {[info procs _expr(OldLet)] == ""} {catch {rename let _expr(OldLet)}}

proc Math [info args _expr(Expr)] [info body _expr(Expr)]
proc _expr(Math) {args} {
  global _expr
  if $_expr(useOldLet) {return [uplevel expr $args]}
  return [uplevel Math $args]
}

proc let {var args} {
  global _expr
  if {[string index $var 0] == "("} {
    while {$args != "" && ![_expr(Complete) $var]} {
      lappend var [lindex $args 0]
      set args [lrange $args 1 end]
    }
    if {![_expr(Complete) $var]} \
      {return -code error "Incomplete variable vector"}
    regsub -all {\(} $var {[list } var
    regsub -all {\)} $var {]} var
    regsub -all {,} $var { } var
    set var [lindex [eval "list $var"] 0]
  }
  if {$args == ""} {return [uplevel [_expr(Vars) $var]]} else {
    if $_expr(useOldLet) {return [uplevel [list _expr(OldLet) $var] $args]}
    if {[lindex $args 0] == "="} {set args [lrange $args 1 end]}
    set value [uplevel _expr(Expr) $args]
    uplevel [list _expr(Let) $var $value]
    return $value
  }
}

proc _expr(Let) {var val {error 1}} {
  if {[llength $var] == 1} {uplevel [list set $var $val]; return}
  if {$error && [llength $var] != [llength $val]} \
    {return -code error "Incorrect number of assignments"}
  set i 0
  foreach name $var {
    if {[llength $name] == 1} {uplevel [list set $name [lindex $val $i]]} \
      else {uplevel [list _expr(Let) $name [lindex $val $i]]}
    incr i
  }
}

proc _expr(Complete) {line} {
  if ![info complete $line] {return 0}
  return [expr [regsub -all {\(} $line {} x] == [regsub -all {\)} $line {} x]]
}

proc _expr(Pull) {lvar} {
  upvar $lvar list
  set v [lindex $list 0]; set list [lrange $list 1 end]
  if {[string index $v 0] == "("} {
    while {$list != "" && ![_expr(Complete) $v]} {
      lappend v [lindex $list 0]
      set list [lrange $list 1 end]
    }
  }
  return $v
}

proc _expr(Vars) {vars} {
  set list {}
  if {[llength $vars] == 1} {return [list set $vars]}
  foreach v $vars {
    if {[llength $v] > 1} {lappend list "\[[_expr(Vars) $v]\]"} \
      else {lappend list "\${$v}"}
  }
  return "list [join $list]"
}


proc _expr(List) {list {level 1}} {
  global _expr
  incr level 1
  if $_expr(useOldLet) {
    set result {}
    foreach e [uplevel $level [list subst $list]] \
     {lappend result [uplevel $level expr $e]}
  } else {
    set result [uplevel $level [list _expr(Expr) $list]]
  }
  return $result
}

