#
#  Binds variables to values in a list (like calling a procedure does).
#

proc _bind(Values) {vars list {name ""}} {
  if {[lindex $vars end] == "args"} {
    uplevel [list set args [lrange $list [expr [llength $vars] - 1] end]]
    set list [lrange $list 0 [expr [llength $vars] - 2]]
    set vars [lrange $vars 0 [expr [llength $vars] - 2]]
  }
  foreach v $vars {
    if {[llength $v] == 1} {
      if {[llength $list] == 0} {
	if {$name != ""} {set name " to \"$name\""}
	return -code error "no value given for parameter \"$v\"$name"
      }
      uplevel [list set $v [lindex $list 0]]
    } else {
      if {[llength $list] == 0} {uplevel set $v} \
	  else {uplevel [list set [lindex $v 0] [lindex $list 0]]}
    }
    set list [lrange $list 1 end]
  }
  if {[llength $list] > 0} {
    if {$name != ""} {set name "called \"$name\" with "}
    return -code error "${name}too many arguments"
  }
}


#
#  Binds flags to variables
#
#  list  =  list of bindings of the form:
#              -flag          (sets a boolean flag
#              -noflag        (unsets a boolean flag)
#              -flag value    (flag requiring a value)
#
#  legal =  list of legal flags in the form:
#
#              flag                a boolean flag
#              {flag init}         a boolean flag with initial value
#              {flag init val}     a boolean with initial and flag values
#              !flag               a negatable boolean
#              {!flag i [v]}       ... with initialization
#              {!flag no yes def}  a negatable flag with value for -noflag,
#                                    -flag and default value
#              !!flag              a negatable boolean initially set
#              {!!flag i [v]}      ... with initialization
#              {!!flag no yes def} ... or with values for -noflag, -flag,
#                                      and default
#              flag=               a flag requiring a value
#              {flag= i}           ... with a default value
#
#  All legal flag values are associated with variables in the calling
#  procedure.  These variabels will be set to either a default or a value
#  indicating that the flag is set.
#

proc _bind(Flags) {legal list {proc ""}} {
  foreach v $legal {
    set v [concat $v [lrange {x 0 1} [llength $v] end]]
    set name [lindex $v 0]
    set default [lindex $v 1]
    set value   [lindex $v 2]
    set init    [lindex [concat $v $default] 3]
    set negatable [regsub {^!} $name {} name]
    set needsvalue [regsub {=$} $name {} name]
    if {$negatable} {
      if [regsub {^!} $name {} name] {set init $value}
      set _legal(no$name) [list $name $default]
      set _needsvalue(no$name) $needsvalue
    }
    uplevel [list set $name $init]
    set _legal($name) [list $name $value]
    set _needsvalue($name) $needsvalue
  }
  if {$proc != ""} {set proc " for \"$proc\""}
  while {[llength [lindex $list 0]] == 1 && \
	     [regexp {^-[^0-9\.]} [lindex $list 0]]} {
    set name [string range [lindex $list 0] 1 end]
    set list [lrange $list 1 end]
    if ![info exists _legal($name)] \
	{return -code error "unknown flag '$name'$proc"}
    set value [lindex $_legal($name) 1]
    if {$_needsvalue($name)} {
      if {[llength $list] == 0 || [regexp {^-[^0-9\.]} [lindex $list 0]]} \
	  {return -code error "flag '$name' requires a value$proc"}
      set value [lindex $list 0]
      set list [lrange $list 1 end]
    }
    uplevel [list set [lindex $_legal($name) 0] $value]
  }
  return $list
}


proc _bind(All) {flags vars list {name ""}} {
  set list [uplevel [list _bind(Flags) $flags $list $name]]
  uplevel [list _bind(Values) $vars $list $name]
}

proc _bind(Rest) {vars flags list {name ""}} {
  for {set i 0; set flist {}} {$i < [llength $list]} {incr i} {
    if {[llength [lindex $list $i]] == 1 && \
	    [regexp {^-[^0-9\.]} [lindex $list $i]]} {
      set flist [lrange $list $i end]
      set list [lrange $list 0 [expr $i-1]]
      break
    }
  }
  uplevel [list _bind(Values) $vars $list $name]
  if {[llength [uplevel [list _bind(Flags) $flags $flist $name]]] > 0} {
    if {$name != ""} {set name "called \"$name\" with "}
    return -code error "${name}too many arguments"
  }
}

#
#  Bind's variables to the values in a list and returns an error code
#

proc _Bind {vars list} {
  return [expr [catch [uplevel [list _bind(Values) $vars $list]]] != 1]
}
