//  IDVI 1.0 source copyright 1996 Garth A. Dickie
//
//  This source is free for non-commercial use.  No warranty, etc.
//  Please acknowledge reuse by including the line:
//
//  "Based in part on IDVI 1.0 source copyright 1996 Garth A. Dickie"
//
//  in your documentation and source code.  For commercial use or
//  distribution, please contact the author.  Please also send
//  questions, comments, bug reports, or fixes.
//
//  Best Regards,
//  Garth A. Dickie
//  dickie@elastic.avid.com

package ibook.v10.idvi.split;

import java.awt.Component;
import java.awt.Image;
import java.io.IOException;
import java.io.FileNotFoundException;
import java.net.URL;
import java.util.Hashtable;

import ibook.v10.idvi.IDVI;
import ibook.v10.idvi.MessageContext;
import ibook.v10.idvi.dvi.DVIFormatException;
import ibook.v10.parameter.ParameterApplication;
import ibook.v10.parameter.ParameterProcessor;
import ibook.v10.parameter.ParameterException;
import ibook.v10.parameter.ParameterMissingException;

public class DVISplitApplication extends ParameterApplication implements MessageContext {
    public static void main( String[ ] args ) {
        DVISplitApplication application = new DVISplitApplication( );
        
        application.init( args );
        application.start( );
    }

    private int columns = 80;

    public void start( ) {
        String name = "java " + getClass( ).getName( );

        try {
            name = parameter.getString( "name", name );
            columns = parameter.getInteger( "columns", columns );

            IDVI.setDebugParameters( parameter );

            Splitter splitter = new Splitter( parameter, this );

            splitter.readDVI( );
            splitter.writeDVI( );
            splitter.writeHTML( );
            splitter.writeImages( );
            splitter.listFonts( );

            showMessage( "" );
        } catch( ParameterMissingException e ) {
            if( ! e.getParameterName( ).equals( "" ))
                System.out.println( name + ": " + e.getMessage( ));

            printUsage( name );
        } catch( ParameterException e ) {
            System.out.println( name + ": " + e.getMessage( ));
        } catch( FileNotFoundException e ) {
            System.out.println( name + ": file \"" + e.getMessage( ) + "\" not found." );
        } catch( DVIFormatException e ) {
            System.out.println( name + ": " + e.getMessage( ));
        } catch( IOException e ) {
            System.out.println( name + ": " + e );
            e.printStackTrace( );
        }
    }

    public void printUsage( String name ) {
        System.out.println( );
        System.out.println( "Use \"" + name + "\" to prepare dvi files for presentation on the web." );
        System.out.println( );
        System.out.println( "Give the name of a single dvi file on the command line, along with any of:" );
        System.out.println( "    -centered     <boolean> Center text on page      (default  true           )" );
        System.out.println( "    -centerPage   <number>  Perfectly centered page  (default  none           )" );
        System.out.println( "    -scaleMin     <number>  Smallest scale factor    (default  1              )" );
        System.out.println( "    -scaleMax     <number>  Largest scale factor     (default  5              )" );
        System.out.println( "    -scaleDefault <number>  Default scale factor     (default  4              )" );
        System.out.println( "    -pageFirst    <number>  Number of first page     (default  1              )" );
        System.out.println( "    -pageDefault  <number>  Page to show by default  (default  1              )" );
        System.out.println( "    -translate    <list>    Translation for URLs     (default  none           )" );
        System.out.println( );
        System.out.println( "    -title        <string>  Document title           (default \"DVI Document\"  )" );
        System.out.println( "    -titleTemplate<string>  Pattern for page title   (default '$title - $page')" );
        System.out.println( "    -noAppletText <string>  Text for old browsers    (default \"Sorry, your...\")" );
        System.out.println( "    -templateFile <filename>Name of model HTML file  (   see documentation    )" );
        System.out.println( );
        System.out.println( "    -backgroundColor   <color>  Background color     (default \"White\"         )" );
        System.out.println( "    -foregroundColor   <color>  Text color           (default \"Black\"         )" );
        System.out.println( "    -linkColor         <color>  Normal link color    (default \"#0000EE\"       )" );
        System.out.println( "    -selectedLinkColor <color>  Tracking link color  (default \"Red\"           )" );
        System.out.println( );
        System.out.println( "    -index        <string>  Name of default page     (default \"index.html\"    )" );
        System.out.println( "    -prefix       <string>  Prefix used in filenames (default \"\"              )" );
        System.out.println( "    -pagePrefix   <string>  Prefix used in filenames (default \"page\"          )" );
        System.out.println( "    -scalePrefix  <string>  Prefix used in filenames (default \"scale\"         )" );
        System.out.println( );
        System.out.println( "Examples:" );
        System.out.println( "    " + name + " paper.dvi -title \"A Paper\"" );
        System.out.println( "    " + name + " small.dvi -title \"A 10pt Report\" -scaleDefault 3 -scaleMax 4" );
        System.out.println( "    " + name + " book.dvi -title \"A Book\" -pageFirst 0 -centerPage 0" );
        System.out.println( );
        System.out.println( "See <http://www.geom.umn.edu/java/idvi/userguide/> for full documentation." );
        System.out.println( );
    }




    //  implementation of the interface MessageContext:

    private String  previousMessage = null;
    private int     previousIndex = 0;
    private boolean wrapped = false;
    private int     usedLength = 0;

    public void showMessage( String message ) {
        if( message.startsWith( "Loading font " ))
            return;

        int messageLength = message.length( );
        int spaceIndex = message.lastIndexOf( ' ' );

        if( previousMessage != null ) {
            boolean matches = spaceIndex == previousIndex &&
                ( spaceIndex == -1 || message.regionMatches( 0, previousMessage, 0, spaceIndex ));

            if( ! matches ) {
                if( wrapped ) System.out.println( );
                System.out.println( );
                System.out.print( message );
                System.out.flush( );

                wrapped = false;
                usedLength = messageLength;
            } else if( usedLength + messageLength - spaceIndex < columns ) {
                System.out.print( ' ' + message.substring( spaceIndex + 1 ));
                System.out.flush( );

                usedLength += messageLength - spaceIndex;
            } else {
                System.out.println( );
                System.out.print( message );
                System.out.flush( );

                wrapped = true;
                usedLength = messageLength;
            }
        } else {
            System.out.print( message );
            System.out.flush( );

            usedLength = messageLength;
        }
        previousMessage = message;
        previousIndex = spaceIndex;
    }
}
